/* Um interpretador Little C. */

#include <stdio.h>
#include <setjmp.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#define NUM_FUNC        100
#define NUM_GLOBAL_VARS 100
#define NUM_LOCAL_VARS  200
#define NUM_BLOCK       100
#define ID_LEN          31
#define FUNC_CALLS      31
#define NUM_PARAMS      31
#define PROG_SIZE       10000
#define LOOP_NEST       31

enum tok_types {DELIMITER, IDENTIFIER, NUMBER, KEYWORD, TEMP,
                STRING, BLOCK};

/* adicione outros tokens C aqui */
enum tokens {ARG, CHAR, INT, IF, ELSE, FOR, DO, WHILE,
             SWITCH, RETURN, EOL, FINISHED, END};

/* adicione outros operadores duplos (tais como ->) aqui */
enum double_ops {LT=1, LE, GT, GE, EQ, NE};

/* Estas sao as constantes usadas para chamar sntx_err() quando
   ocorre um erro de sintaxe. Adicione mais se desejar.
   NOTA: SYNTAX  uma mensagem genrica de erro usada quando
   nenhuma outra parece apropriada.
 */
enum error_msg
     {SYNTAX, UNBAL_PARENS, NO_EXP, EQUALS_EXPECTED,
      NOT_VAR, PARAM_ERR, SEMI_EXPECTED,
      UNBAL_BRACES, FUNC_UNDEF, TYPE_EXPECTED,
      NEST_FUNC, RET_NOCALL, PAREN_EXPECTED,
      WHILE_EXPECTED, QUOTE_EXPECTED, NOT_TEMP,
      TOO_MANY_LVARS};

char *prog; /* posiao corrente no cdigo fonte */
char *p_buf; /* aponta para o incio da
                rea de carga do programa */
jmp_buf e_buf; /* mantm ambiente para longjmp() */

/* Uma matriz destas estruturas manter a informaao
   associada com as variveis globais.
*/
struct var_type {
  char var_name[ID_LEN];
  enum variable_type var_type;
  int value;
} global_vars[NUM_GLOBAL_VARS];

struct var_type local_var_stack[NUM_LOCAL_VARS];

struct func_type {
  char func_name[ID_LEN];
  char *loc; /* posiao do ponto de entrada da funao no arquivo */
} func_table[NUM_FUNC];

int call_stack[NUM_FUNC];

struct commands { /* Tabela de busca de palavras reservadas */
  char command[20];
  char tok;
} table[] = { /* Comandos devem ser escritos */
  "if", IF,   /* em minscula nesta tabela */
  "else", ELSE,
  "for", FOR,
  "do", DO,
  "while", WHILE,
  "char", CHAR,
  "int", INT,
  "return", RETURN,
  "end", END,
  "", END  /* marca fim da tabela */
};

char token[80];
char token_type, tok;

int functos; /* ndice para o topo da pilha de chamadas de funao */
int func_index; /* ndice na tabela de funoes */
int gvar_index; /* ndice na tabela global de variveis */
int lvartos; /* ndice para a pilha de variveis locais */

int ret_value; /* valor de retorno de funao */

void print(void), prescan(void);
void decl_global(void), call(void), putback(void);
void decl_local(void), local_push(struct var_type i);
void eval_exp(int *value), sntx_err(int error);
void exec_if(void), find_eob(void), exec_for(void);
void get_params(void), get_args(void);
void exec_while(void), func_push(int i), exec_do(void);
void assign_var(char *var_name, int value);
int load_program(char *p, char *fname), find_var(char *s);
void interp_block(void), func_ret(void);
int func_pop(void), is_var(char *s), get_token(void);

char *find_func(char *name);

main(int argc, char *argv[])
{
  if(argc!=2) {
    printf("Uso: littlec <nome_de_arquivo>\n");
    exit(1);
  }

  /* aloca memria para o programa */
  if((p_buf=(char *) malloc(PROG_SIZE))==NULL) {
    printf("Falha de alocaao");
    exit(1);
  }

  /* carrega o programa a executar */
  if(!load_program(p_buf, argv[1])) exit(1);

  if(setjmp(e_buf)) exit(1); /* inicializa buffer de longjmp */

  /* define apontador de programa para o incio do buffer */
  prog = p_buf;
  prescan(); /* procura a posiao de todas as funoes
                e variveis globais no programa */
  gvar_index = 0;  /* inicializa ndice de variveis globais */
  lvartos = 0;     /* inicializa ndice da pilha de variveis locais */
  functos = 0;     /* inicializa o ndice da pilha de CALL */

/* prepara chamada de main */
  prog = find_func("main"); /* acha o ponto de incio do programa */
  prog--; /* volta para { inicial */
  strcpy(token, "main");
  call();  /* inicia interpretaao */
  return 0;
}

/* Interpreta um nico comando ou bloco de cdigo. Quando
   interp_block() retorna da sua chamada inicial, a chave
   final (ou um return) foi encontrada em main().
*/
void interp_block(void)
{
  int value;
  char block = 0;

  do {
    token_type = get_token();

    /* Se interpretando um nico comando,
       retorne ao achar o primeiro ponto e vrgula
    */

    /* Veja que tipo de token est pronto */
    if(token_type==IDENTIFIER) {
      /* Nao  uma palavra reservada, logo processa expressao. */
      putback(); /* devolve token para a entrada para
                    posterior processamento por eval_exp() */
      eval_exp(&value);
      if(*token!=';') sntx_err(SEMI_EXPECTED);
    }
    else if(token_type==BLOCK) { /* se delimitador de bloco */
      if(*token=='{') /*  um bloco */
        block = 1; /* intepretando bloco, nao comando */
      else return;
    }
    else /*  palavra reservada */
      switch(tok) {
        case CHAR:
        case INT:       /* declara variveis locais */
          putback();
          decl_local();
          break;
        case RETURN:    /* retorna da chamada de funao */
          func_ret();
          return;
        case IF:        /* processa um comando if */
          exec_if();
          break;
        case ELSE:      /* processa um comando else */
          find_eob();   /* acha fim do bloco de else
                           e continua execuao */
          break;
        case WHILE:     /* processa uma repetiao while */
          exec_while();
          break;
        case DO:        /* processa uma repetiao do-while */
          exec_do();
          break;
        case FOR:       /* processa uma repetiao for */
          exec_for();
          break;
        case END:
          exit(0);
      }
  } while (tok != FINISHED && block);
}

/* Carrega um programa. */
load_program(char *p, char *fname)
{
  FILE *fp;
  int i=0;

  if((fp=fopen(fname, "rb"))==NULL) return 0;

  i = 0;
  do {
    *p = getc(fp);
    p++; i++;
  } while(!feof(fp) && i<PROG_SIZE);
  if(*(p-2)==0x1a) *(p-2) = '\0'; /* encerra o programa com nulo */
  else *(p-1) = '\0';
  fclose(fp);
  return 1;
}

/* Acha a posiao de todas as funoes no programa
   e armazena todas as variveis globais. */
void prescan(void)
{
  char *p;
  char temp[32];
  int brace = 0; /* Quando 0, esta varivel ns indica que a
                    posiao corrente no cdigo fonte est fora
                    de qualquer funao. */

  p = prog;
  func_index = 0;
  do {
    while(brace) {  /* deixa de lado o cdigo dentro das funoes */
      get_token();
      if(*token=='{') brace++;
      if(*token=='}') brace--;
    }

    get_token();

    if(tok==CHAR || tok==INT) { /*  uma varivel global */
      putback();
      decl_global();
    }
    else if(token_type==IDENTIFIER) {
       strcpy(temp, token);
       get_token();
       if(*token=='(') { /* tm que ser uma funao */
         func_table[func_index].loc = prog;
         strcpy(func_table[func_index].func_name, temp);
         func_index++;
         while(*prog!=')') prog++;
         prog++;
         /* agora prog aponta para o abre chaves da funao */
       }
       else
         putback();
    }
    else if(*token=='{') brace++;
  } while(tok!=FINISHED);
  prog = p;
}

char *find_func(char *name)
{
  register int i;

  for(i=0; i<func_index; i++)
    if(!strcmp(name, func_table[i].func_name))
      return func_table[i].loc;

  return NULL;
}

/* Declara uma varivel global. */
void decl_global(void)
{
  get_token();  /* obtm o tipo */
  global_vars[gvar_index].var_type = tok;
  global_vars[gvar_index].value = 0; /* inicializa com 0 */

  do { /* processa lista separada por vrgulas */
    get_token();  /* obtm nome */
    strcpy(global_vars[gvar_index].var_name, token);
    get_token();
    gvar_index++;
  } while(*token==',');
  if(*token!=';') sntx_err(SEMI_EXPECTED);
}

/* Declara uma varivel local. */
void decl_local(void)
{
  struct var_type i;

  get_token();  /* obtm tipo */

  i.var_type = tok;
  i.value = 0;  /* inicializa com 0 */

  do { /* processa lista separada por vrgulas */
    get_token();  /* obtm nome da varivel */
    strcpy(i.var_name, token);
    local_push(i);
    get_token();
    gvar_index++;
  } while(*token==',');
  if(*token!=';') sntx_err(SEMI_EXPECTED);
}

/* Chama una funao. */
void call(void)
{
  char *loc, *temp;
  int lvartemp;

  loc = find_func(token); /* encontra ponto de entrada da funao */
  if(loc==NULL)
    sntx_err(FUNC_UNDEF); /* funao nao definida */
  else {
    lvartemp = lvartos;  /* guarda ndice da pilha de var locais */
    get_args();  /* obtm argumentos da funao */
    temp = prog; /* salva endereo de retorno */
    func_push(lvartemp); /* salva ndice da pilha de var locais */
    prog = loc;  /* redefine prog para o incio da funao */
    get_params(); /* carrega os parmetros da funao com
                     os valores dos argumentos */
    interp_block(); /* interpreta a funao */
    prog = temp; /* redefine o apontador de programa */
    lvartos = func_pop(); /* redefine a pilha de var locais */
  }
}

/* Empilha os argumentos de uma funao na pilha de variveis
   locais. */
void get_args(void)
{
  int value, count, temp[NUM_PARAMS];
  struct var_type i;

  count = 0;
  get_token();
  if(*token!='(') sntx_err(PAREN_EXPECTED);

  /* processa uma lista de valores separados por vrgulas */
  do {
    eval_exp(&value);
    temp[count] = value;
    get_token();
    count++;
  }while(*token==',');
  count--;
  /* agora, empilha em local_var_stack na ordem invertida */
  for(; count>=0; count--) {
    i.value = temp[count];
    i.var_type = ARG;
    local_push(i);
  }
}

/* Obtm parmetros da funao. */
void get_params(void)
{
  struct var_type *p;
  int i;

  i=lvartos-1;
  do { /* processa lista de parmetros separados por vrgulas */
    get_token();
    p = &local_var_stack[i];
    if(*token!=')') {
      if(tok!=INT && tok != CHAR) sntx_err(TYPE_EXPECTED);
      p->var_type = token_type;
      get_token();

      /* liga nome do parmetro com argumento que j est na
         pilha de variveis locais */
      strcpy(p->var_name, token);
      get_token();
      i--;
    }
    else break;
  } while(*token==',');
  if(*token!=')') sntx_err(PAREN_EXPECTED);
}

/* Retorna de uma funao. */
void func_ret(void)
{
  int value;

  value = 0;
  /* obtm valor de retorno, se houver */
  eval_exp(&value);

  ret_value = value;
}

/* Empilha uma varivel local. */
void local_push(struct var_type i)
{
  if(lvartos>NUM_LOCAL_VARS)
    sntx_err(TOO_MANY_LVARS);

  local_var_stack[lvartos] = i;
  lvartos++;
}

/* Desempilha ndice na pilha de variveis locais */
func_pop(void)
{
  functos--;
  if(functos<0) sntx_err(RET_NOCALL);
  return(call_stack[functos]);
}

/* Empilha ndice da pilha de variveis locais */
void func_push(int i)
{
  if(functos>NUM_FUNC)
    sntx_err(NEST_FUNC);
  call_stack[functos] = i;
  functos++;
}

/* Atribui um valor a uma varivel. */
void assign_var(char *var_name, int value)
{
  register int i;

  /* primeiro, veja se  uma varivel local */
  for (i=lvartos-1; i>=call_stack[functos-1]; i--)  {
    if(!strcmp(local_var_stack[i].var_name, var_name)) {
      local_var_stack[i].value = value;
      return;
    }
  }
  if(i < call_stack[functos-1])
  /* se nao  local, tente na tabela de variveis globais */
    for(i=0; i<NUM_GLOBAL_VARS; i++)
      if(!strcmp(global_vars[i].var_name, var_name)) {
        global_vars[i].value = value;
        return;
      }
  sntx_err(NOT_VAR); /* varivel nao encontrada */
}

/* Encontra o valor de uma varivel. */
int find_var(char *s)
{
  register int i;

   /* primeiro, veja se  uma varivel local */
  for (i=lvartos-1; i>=call_stack[functos-1]; i--)
    if(!strcmp(local_var_stack[i].var_name, token))
      return local_var_stack[i].value;

  /* se nao  local, tente na tabela de variveis globais */
  for(i=0; i<NUM_GLOBAL_VARS; i++)
    if(!strcmp(global_vars[i].var_name, s))
      return global_vars[i].value;

  sntx_err(NOT_VAR); /* varivel nao encontrada */
}

/* Determina se um identificador  uma varivel.
   Retorna 1 se a varivel  encontrada, 0 caso contrrio.
*/
int is_var(char *s)
{
  register int i;

  /* primeiro, veja se  uma varivel local */
  for (i=lvartos-1; i>=call_stack[functos-1]; i--)
    if(!strcmp(local_var_stack[i].var_name, token))
      return 1;

  /* caso contrrio, tente com as variveis globais */
  for(i=0; i<NUM_GLOBAL_VARS; i++)
    if(!strcmp(global_vars[i].var_name, s))
      return 1;

  return 0;
}

/* Executa um comando if. */
void exec_if(void)
{
  int cond;

  eval_exp(&cond); /* obtm expressao esquerda */

  if(cond) { /*  verdadeira, portanto processa alvo do IF */
    interp_block();
  }
  else { /* caso contrrio ignore o bloco de IF e
            processe o ELSE, se existir */
    find_eob();
    get_token();

    if(tok!=ELSE) {
      putback(); /* devolve o token se nao  ELSE */
      return;
    }
    interp_block();
  }
}

/* Executa uma repetiao while. */
void exec_while(void)
{
  int cond;
  char *temp;

  putback();
  temp = prog;  /* salva posiao do incio da repetiao while */
  get_token();
  eval_exp(&cond);  /* verifica a expressao condicional */
  if(cond) interp_block();  /* se verdadeira, interpreta */
  else {  /* caso contrrio, ignore a repetiao */
    find_eob();
    return;
  }
  prog = temp; /* volta para o incio da repetiao */
}

/* Executa uma repetiao do. */
void exec_do(void)
{
  int cond;
  char *temp;

  putback();
  temp = prog; /* salva posiao do incio da repetiao do */

  get_token(); /* obtm incio da repetiao */
  interp_block(); /* interpreta a repetiao */
  get_token();
  if(tok!=WHILE) sntx_err(WHILE_EXPECTED);
  eval_exp(&cond); /* verifica a condiao da repetiao */
  if(cond) prog = temp; /* se verdadeiro, repete;
                           caso contrrio, continua */
}

/* Acha o fim de um bloco. */
void find_eob(void)
{
  int brace;

  get_token();
  brace = 1;
  do {
    get_token();
    if(*token=='{') brace++;
    else if(*token=='}') brace--;
  } while(brace);
}

/* Executa uma repetiao for. */
void exec_for(void)
{
  int cond;
  char *temp, *temp2;
  int brace ;

  get_token();
  eval_exp(&cond);  /* inicializa a expressao */
  if(*token!=';') sntx_err(SEMI_EXPECTED);
  prog++; /* passa do ; */
  temp = prog;
  for(;;) {
    eval_exp(&cond);  /* verifica a condiao */
    if(*token!=';') sntx_err(SEMI_EXPECTED);
    prog++; /* passa do ; */
    temp2 = prog;

    /* acha o incio do bloco do for */
    brace = 1;
    while(brace) {
      get_token();
      if(*token=='(') brace++;
      if(*token==')') brace--;
    }

    if(cond) interp_block();  /* se verdadeiro, interpreta */
    else {  /* caso contrrio, ignora a repetiao */
      find_eob();
      return;
    }
    prog = temp2;
    eval_exp(&cond); /* efetua o incremento */
    prog = temp; /* volta para o incio */
  }
}

