/*
 * QUOTA    An implementation of the diskquota system for the LINUX operating
 *          system. QUOTA is implemented using the BSD systemcall interface as
 *          the means of communication with the user level. Should work for all
 *          filesystems because of integration into the VFS layer of the
 *          operating system. This is based on the Melbourne quota system wich
 *          uses both user and group quota files.
 * 
 *          Main layer of quota management
 * 
 * Version: $Id: dquot.c,v 3.3 1994/01/09 10:45:23 mvw Exp mvw $
 * 
 * Authors: Marco van Wieringen <v892273@si.hhs.nl> <mvw@mcs.nl.mugnet.org>
 *          Edvard Tuinder <v892231@si.hhs.nl> <ed@delirium.nl.mugnet.org>
 *          Linus Torvalds <Linus.Torvalds@cc.helsinki.FI>
 * 
 *          This program is free software; you can redistribute it and/or
 *          modify it under the terms of the GNU General Public License as
 *          published by the Free Software Foundation; either version 2 of the
 *          License, or (at your option) any later version.
 */

#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/config.h>

#ifdef CONFIG_QUOTA
#include <linux/types.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/segment.h>
#include <linux/tty.h>
#include <linux/malloc.h>
#include <linux/mount.h>
#include <linux/fileio.h>

#include <asm/segment.h>
#include <sys/sysmacros.h>

#define DEBUG_QUOTA 0 /* Level of debugging the higher the more stuff you get */

static char quotamessage[MAX_QUOTA_MESSAGE];
static char *quotatypes[] = INITQFNAMES;

static int nr_dquots = 0, nr_free_dquots = 0;
static struct dquot *hash_table[NR_DQHASH];
static struct dquot *first_dquot;

static struct wait_queue *dquot_wait = (struct wait_queue *)NULL;

extern void add_dquot_ref(dev_t dev, short type);
extern void reset_dquot_ptrs(dev_t dev, short type);
extern void close_filp(struct file *filp, int fd);

#ifndef min
#define min(a,b) ((a) < (b)) ? (a) : (b)
#endif

/*
 * Functions for management of the hashlist.
 */
static inline int const hashfn(dev_t dev, unsigned int id, short type)
{
   return ((dev ^ id) * (MAXQUOTAS - type)) % NR_DQHASH;
}

static inline struct dquot **const hash(dev_t dev, unsigned int id, short type)
{
   return hash_table + hashfn(dev, id, type);
}

static void insert_dquot_free(struct dquot *dquot)
{
   dquot->dq_next = first_dquot;
   dquot->dq_prev = first_dquot->dq_prev;
   dquot->dq_next->dq_prev = dquot;
   dquot->dq_prev->dq_next = dquot;
   first_dquot = dquot;
}

static void remove_dquot_free(struct dquot *dquot)
{
   if (first_dquot == dquot)
      first_dquot = first_dquot->dq_next;
   if (dquot->dq_next)
      dquot->dq_next->dq_prev = dquot->dq_prev;
   if (dquot->dq_prev)
      dquot->dq_prev->dq_next = dquot->dq_next;
   dquot->dq_next = dquot->dq_prev = NODQUOT;
}

static void insert_dquot_hash(struct dquot *dquot)
{
   struct dquot **h;

   h = hash(dquot->dq_dev, dquot->dq_id, dquot->dq_type);
   dquot->dq_hash_next = *h;
   dquot->dq_hash_prev = NODQUOT;
   if (dquot->dq_hash_next)
      dquot->dq_hash_next->dq_hash_prev = dquot;
   *h = dquot;
}

static void remove_dquot_hash(struct dquot *dquot)
{
   struct dquot **h;

   h = hash(dquot->dq_dev, dquot->dq_id, dquot->dq_type);
   if (*h == dquot)
      *h = dquot->dq_hash_next;
   if (dquot->dq_hash_next)
      dquot->dq_hash_next->dq_hash_prev = dquot->dq_hash_prev;
   if (dquot->dq_hash_prev)
      dquot->dq_hash_prev->dq_hash_next = dquot->dq_hash_next;
   dquot->dq_hash_prev = dquot->dq_hash_next = NODQUOT;
}

static void put_last_free(struct dquot *dquot)
{
   remove_dquot_free(dquot);
   dquot->dq_prev = first_dquot->dq_prev;
   dquot->dq_prev->dq_next = dquot;
   dquot->dq_next = first_dquot;
   dquot->dq_next->dq_prev = dquot;
}

static void grow_dquots(void)
{
   struct dquot *dquot;
   int i;

   if (!(dquot = (struct dquot*) get_free_page(GFP_KERNEL)))
      return;
   i = PAGE_SIZE / sizeof(struct dquot);
   nr_dquots += i;
   nr_free_dquots += i;
   if (!first_dquot)
      dquot->dq_next = dquot->dq_prev = first_dquot = dquot++, i--;
   for ( ; i ; i-- )
      insert_dquot_free(dquot++);
}

/*
 * Functions for locking and waiting on dquots.
 */
static void __wait_on_dquot(struct dquot *dquot)
{
   struct wait_queue wait = {current, NULL};

   add_wait_queue(&dquot->dq_wait, &wait);
   current->state = TASK_UNINTERRUPTIBLE;
   while (dquot->dq_flags & DQ_LOCKED) {
      dquot->dq_flags |= DQ_WANT;
      schedule();
      current->state = TASK_UNINTERRUPTIBLE;
   }
   remove_wait_queue(&dquot->dq_wait, &wait);
   current->state = TASK_RUNNING;
}

static inline void wait_on_dquot(struct dquot *dquot)
{
   if (dquot->dq_flags & DQ_LOCKED)
      __wait_on_dquot(dquot);
}

static inline void lock_dquot(struct dquot *dquot)
{
   wait_on_dquot(dquot);
   dquot->dq_flags |= DQ_LOCKED;
}

static inline void unlock_dquot(struct dquot *dquot)
{
   dquot->dq_flags &= ~DQ_LOCKED;
   if (dquot->dq_flags & DQ_WANT) {
      dquot->dq_flags &= ~DQ_WANT;
      wake_up(&dquot->dq_wait);
   }
}
/*
 * Note that we don't want to disturb any wait-queues when we discard
 * an dquot.
 *
 * FIXME: As soon as we have a nice solution for the inode problem we
 *        can also fix this one. I.e. the volatile part.
 */
static void clear_dquot(struct dquot * dquot)
{
   struct wait_queue *wait;

   wait_on_dquot(dquot);
   remove_dquot_hash(dquot);
   remove_dquot_free(dquot);
   wait = ((volatile struct dquot *) dquot)->dq_wait;
   if (dquot->dq_count)
      nr_free_dquots++;
   memset(dquot,0,sizeof(*dquot));
   ((volatile struct dquot *) dquot)->dq_wait = wait;
   insert_dquot_free(dquot);
}

/*
 * Check if there are dquot locked on a filesystem that needs
 * to be removed.
 */
static int quotas_busy(dev_t dev, short type)
{
   struct dquot *dquot;
   int i;

   dquot = first_dquot;
   for (i = 0 ; i < nr_dquots ; i++, dquot = dquot->dq_next) {
      if (!dquot->dq_count || dquot->dq_dev != dev)
         continue;
      if (type != -1 && dquot->dq_type != type)
         continue;
      if (!(dquot->dq_flags & DQ_LOCKED))
         continue;
      return 0;
   }
   return 1;
}

static void write_dquot(struct dquot *dquot)
{
   short type = dquot->dq_type;
   struct file *filp = dquot->dq_mnt->mnt_quotas[type];
   unsigned short fs;

   if (!(dquot->dq_flags & DQ_MOD) || (filp == (struct file *)0))
      return;
   if (filp->f_op->lseek) {
      if (filp->f_op->lseek(filp->f_inode, filp, dqoff(dquot->dq_id),0) !=
          dqoff(dquot->dq_id))
         return;
   } else
      filp->f_pos = dqoff(dquot->dq_id);
   lock_dquot(dquot);
   fs = get_fs();
   set_fs(KERNEL_DS);
   if (filp->f_op->write(filp->f_inode, filp, (char *)&dquot->dq_dqb,
       sizeof(struct dqblk)) == sizeof(struct dqblk))
      dquot->dq_flags &= ~DQ_MOD;
   set_fs(fs);
   unlock_dquot(dquot);
}

static void read_dquot(struct dquot *dquot)
{
   short type = dquot->dq_type;
   struct file *filp = dquot->dq_mnt->mnt_quotas[type];
   unsigned short fs;

   if (filp == (struct file *)0)
      return;
   if (filp->f_op->lseek) {
      if (filp->f_op->lseek(filp->f_inode, filp, dqoff(dquot->dq_id),0) !=
          dqoff(dquot->dq_id))
         return;
   } else
      filp->f_pos = dqoff(dquot->dq_id);
   lock_dquot(dquot);
   fs = get_fs();
   set_fs(KERNEL_DS);
   filp->f_op->read(filp->f_inode, filp, (char *)&dquot->dq_dqb, sizeof(struct dqblk));
   set_fs(fs);
   if (dquot->dq_bhardlimit == 0 && dquot->dq_bsoftlimit == 0 &&
       dquot->dq_ihardlimit == 0 && dquot->dq_isoftlimit == 0)
      dquot->dq_flags |= DQ_FAKE;
   else
      dquot->dq_flags &= ~DQ_FAKE;
   unlock_dquot(dquot);
}

int sync_dquots(dev_t dev, short type)
{
   struct dquot *dquot = first_dquot;
   int i;

   for(i = 0; i < nr_dquots * 2; i++, dquot = dquot->dq_next) {
      if (!dquot->dq_count || (dev && dquot->dq_dev != dev))
         continue;
      if (type != -1 && dquot->dq_type != type)
         continue;
      wait_on_dquot(dquot);
      if (dquot->dq_flags & DQ_MOD)
         write_dquot(dquot);
   }
   return 0;
   /* NOTREACHED */
}

/*
 * Trash the cache for a certain type on a device.
 */
void invalidate_dquots(dev_t dev, short type)
{
   struct dquot *dquot, *next;
   int i;

   next = first_dquot;
   for(i = nr_dquots ; i > 0 ; i--) {
      dquot = next;
      next = dquot->dq_next;
      if (dquot->dq_dev != dev || dquot->dq_type != type)
         continue;
      if (dquot->dq_flags & DQ_MOD)
         write_dquot(dquot);
      clear_dquot(dquot);
   }
}

/*
 * Kick the message in quotamessage to the tty.
 */
static void quota_message(void)
{
   struct tty_struct *tty;

   if (current->tty >= 0) {
      tty = TTY_TABLE(current->tty);
      (void) tty_write_data(tty, quotamessage, strlen(quotamessage),
                            (void *)0, (void *)0);
   }
}

/*
 * Check quota for inodes. Returns QUOTA_OK if can allocate and
 * NO_QUOTA if it can't.
 */
static int check_idq(struct dquot *dquot, int id, short type, u_long wanted_inodes)
{
   if (wanted_inodes == 0 || dquot->dq_flags & DQ_FAKE)
      return QUOTA_OK;
   if (dquot->dq_ihardlimit &&
      (dquot->dq_curinodes + wanted_inodes) >= dquot->dq_ihardlimit) {
      if (!(dquot->dq_flags & DQ_INODES)) {
         sprintf(quotamessage,
                "File LIMIT reached on %s for %s %d. !! NO MORE !!\n\r",
                dquot->dq_mnt->mnt_devname, quotatypes[type], id);
         quota_message();
         dquot->dq_flags &= DQ_INODES;
      }
      return NO_QUOTA;
   }
   if (dquot->dq_isoftlimit &&
      (dquot->dq_curinodes + wanted_inodes) >= dquot->dq_isoftlimit &&
       dquot->dq_itime && CURRENT_TIME >= dquot->dq_itime) {
      sprintf(quotamessage,
              "File QUOTA exceeded TOO long on %s for %s %d. !! NO MORE !!\n\r",
              dquot->dq_mnt->mnt_devname, quotatypes[type], id);
      quota_message();
      return NO_QUOTA;
   }
   if (dquot->dq_isoftlimit &&
      (dquot->dq_curinodes + wanted_inodes) >= dquot->dq_isoftlimit &&
       dquot->dq_itime == 0) {
      sprintf(quotamessage,
              "File QUOTA exceeded on %s for %s %d\n\r",
              dquot->dq_mnt->mnt_devname, quotatypes[type], id);
      quota_message();
      dquot->dq_itime = CURRENT_TIME + dquot->dq_mnt->mnt_iexp[type];
   }
   return QUOTA_OK;
   /* NOTREACHED */
}

/*
 * Check quota for blocks. Returns QUOTA_OK if can allocate and
 * NO_QUOTA if it can't. When we can't allocate wanted_blocks you get
 * the number we can allocate in avail_blocks.
 */
static int check_bdq(struct dquot *dquot, int id, short type,
                     u_long wanted_blocks, u_long *avail_blocks)
{
   *avail_blocks = wanted_blocks;
   if (wanted_blocks == 0 || dquot->dq_flags & DQ_FAKE)
      return QUOTA_OK;
   if (dquot->dq_bhardlimit &&
      (dquot->dq_curblocks + wanted_blocks) >= dquot->dq_bhardlimit) {
      if ((dquot->dq_flags & DQ_BLKS) == 0) {
         sprintf(quotamessage,
                 "Block LIMIT reached on %s for %s %d. !! NO MORE !!\n\r",
                 dquot->dq_mnt->mnt_devname, quotatypes[type], id);
         quota_message();
         dquot->dq_flags &= DQ_BLKS;
      }
      if (dquot->dq_curblocks != dquot->dq_bhardlimit) {
         *avail_blocks = dquot->dq_bhardlimit - dquot->dq_curblocks;
         return QUOTA_OK;
      } else
         return NO_QUOTA;
   }
   if (dquot->dq_bsoftlimit &&
      (dquot->dq_curblocks + wanted_blocks) >= dquot->dq_bsoftlimit &&
       dquot->dq_btime && CURRENT_TIME >= dquot->dq_btime) {
      sprintf(quotamessage,
              "Block QUOTA exceeded TOO long on %s for %s %d. !! NO MORE !!\n\r",
              dquot->dq_mnt->mnt_devname, quotatypes[type], id);
         quota_message();
      return NO_QUOTA;
   }
   if (dquot->dq_bsoftlimit &&
      (dquot->dq_curblocks + wanted_blocks) >= dquot->dq_bsoftlimit &&
       dquot->dq_btime == 0) {
      sprintf(quotamessage,
              "Block QUOTA exceeded on %s for %s %d\n\r",
              dquot->dq_mnt->mnt_devname, quotatypes[type], id);
      quota_message();
      dquot->dq_btime = CURRENT_TIME + dquot->dq_mnt->mnt_bexp[type];
   }
   return QUOTA_OK;
   /* NOTREACHED */
}

/*
 * Add a number of inodes and blocks to a diskquota.
 */
static inline void incr_quota(struct dquot *dquot, u_long inodes, u_long blocks)
{
   lock_dquot(dquot);
   dquot->dq_curinodes += inodes;
   dquot->dq_curblocks += blocks;
   dquot->dq_flags |= DQ_MOD;
   unlock_dquot(dquot);
}

/*
 * Remove a number of inodes and blocks from a quota.
 * Reset gracetimes if under softlimit.
 */
static inline void decr_quota(struct dquot *dquot, u_long inodes, u_long blocks)
{
   lock_dquot(dquot);
   if (dquot->dq_curinodes >= inodes)
      dquot->dq_curinodes -= inodes;
   else
      dquot->dq_curinodes = 0;
   if (dquot->dq_curinodes < dquot->dq_isoftlimit)
      dquot->dq_itime = (time_t) 0;
   dquot->dq_flags &= ~DQ_INODES;
   if (dquot->dq_curblocks >= blocks)
      dquot->dq_curblocks -= blocks;
   else
      dquot->dq_curblocks = 0;
   if (dquot->dq_curblocks < dquot->dq_bsoftlimit)
      dquot->dq_btime = (time_t) 0;
   dquot->dq_flags &= ~DQ_BLKS;
   dquot->dq_flags |= DQ_MOD;
   unlock_dquot(dquot);
}

/*
 * Initialize a dquot-struct with new quota info. This is used by the
 * systemcall interface functions.
 */ 
static int set_dqblk(dev_t dev, int id, short type, int flags, struct dqblk *dqblk)
{
   struct dquot *dquot;
   struct dqblk dq_dqblk;
   int error;

   if (dqblk == (struct dqblk *)0)
      return -EFAULT;

   if (flags & QUOTA_SYSCALL) {
      if ((error = verify_area(VERIFY_READ, dqblk, sizeof(struct dqblk))) != 0)
         return error;
      memcpy_fromfs(&dq_dqblk, dqblk, sizeof(struct dqblk));
   } else {
      memcpy(&dq_dqblk, dqblk, sizeof(struct dqblk));
   }
   if ((dquot = dqget(dev, id, type)) != NODQUOT) {
      lock_dquot(dquot);
      if (id > 0 && ((flags & SET_QUOTA) || (flags & SET_QLIMIT))) {
         dquot->dq_bhardlimit = dq_dqblk.dqb_bhardlimit;
         dquot->dq_bsoftlimit = dq_dqblk.dqb_bsoftlimit;
         dquot->dq_ihardlimit = dq_dqblk.dqb_ihardlimit;
         dquot->dq_isoftlimit = dq_dqblk.dqb_isoftlimit;
      }
      if ((flags & SET_QUOTA) || (flags & SET_USE)) {
         if (dquot->dq_isoftlimit &&
             dquot->dq_curinodes < dquot->dq_isoftlimit &&
             dq_dqblk.dqb_curinodes >= dquot->dq_isoftlimit)
            dquot->dq_itime = CURRENT_TIME + dquot->dq_mnt->mnt_iexp[type];
         dquot->dq_curinodes = dq_dqblk.dqb_curinodes;
         if (dquot->dq_curinodes < dquot->dq_isoftlimit)
            dquot->dq_flags &= ~DQ_INODES;
         if (dquot->dq_bsoftlimit &&
             dquot->dq_curblocks < dquot->dq_bsoftlimit &&
             dq_dqblk.dqb_curblocks >= dquot->dq_bsoftlimit)
            dquot->dq_btime = CURRENT_TIME + dquot->dq_mnt->mnt_bexp[type];
         dquot->dq_curblocks = dq_dqblk.dqb_curblocks;
         if (dquot->dq_curblocks < dquot->dq_bsoftlimit)
            dquot->dq_flags &= ~DQ_BLKS;
      }
      if (id == 0) {
         /* 
          * Change in expiretimes, change them in dq_mnt.
          */
         dquot->dq_mnt->mnt_bexp[type] = dquot->dq_btime = dq_dqblk.dqb_btime;
         dquot->dq_mnt->mnt_iexp[type] = dquot->dq_itime = dq_dqblk.dqb_itime;
      }
      if (dq_dqblk.dqb_bhardlimit == 0 && dq_dqblk.dqb_bsoftlimit == 0 &&
          dq_dqblk.dqb_ihardlimit == 0 && dq_dqblk.dqb_isoftlimit == 0)
         dquot->dq_flags |= DQ_FAKE; /* No limits, only usage */
      dquot->dq_flags |= DQ_MOD;
      unlock_dquot(dquot);
      dqput(dquot);
   }
   return 0;
}

static int get_quota(dev_t dev, int id, short type, struct dqblk * dqblk)
{
   struct vfsmount *vfsmnt;
   struct dquot *dquot;
   int error;

   if ((vfsmnt = lookup_vfsmnt(dev)) != (struct vfsmount *)0 &&
        vfsmnt->mnt_quotas[type] != (struct file *)0) {
      if (dqblk == (struct dqblk *) 0)
         return -EFAULT;

      if ((error = verify_area(VERIFY_WRITE, dqblk, sizeof(struct dqblk))) != 0)
         return (error);

      if ((dquot = dqget(dev, id, type)) != NODQUOT) {
         memcpy_tofs(dqblk, (char *)&dquot->dq_dqb, sizeof(struct dqblk));
         dqput(dquot);
         return 0;
      }
      return -ESRCH;
   } else
      return -ESRCH;
}

/*
 * Turn quota off on a device. type == -1 ==> quotaoff for all types (umount)
 */
int quota_off(dev_t dev, short type)
{
   struct vfsmount *vfsmnt;
   short cnt;

   if (type != -1) {
      if ((vfsmnt = lookup_vfsmnt(dev)) == (struct vfsmount *)0 ||
           vfsmnt->mnt_quotas[type] == (struct file *)0)
         return -ESRCH;
      if (!quotas_busy(dev, type))
         return -EBUSY;
      reset_dquot_ptrs(dev, type);
      invalidate_dquots(dev, type);
      close_filp(vfsmnt->mnt_quotas[type], 0);
      vfsmnt->mnt_quotas[type] = (struct file *)0;
      vfsmnt->mnt_iexp[type] = vfsmnt->mnt_bexp[type] = (time_t)0;
   } else {
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         if ((vfsmnt = lookup_vfsmnt(dev)) == (struct vfsmount *)0 ||
              vfsmnt->mnt_quotas[cnt] == (struct file *)0)
            continue;
         if (!quotas_busy(dev, cnt))
            return -EBUSY;
         reset_dquot_ptrs(dev, cnt);
         invalidate_dquots(dev, cnt);
         close_filp(vfsmnt->mnt_quotas[cnt], 0);
         vfsmnt->mnt_quotas[cnt] = (struct file *)0;
         vfsmnt->mnt_iexp[cnt] = vfsmnt->mnt_bexp[cnt] = (time_t)0;
      }
   }
   return 0;
}

static int quota_on(dev_t dev, short type, char *path)
{
   struct file *filp = (struct file *)0;
   struct vfsmount *vfsmnt;
   struct inode *inode;
   struct dquot *dquot;
   char *tmp;
   int error;

   if ((vfsmnt = lookup_vfsmnt(dev)) == (struct vfsmount *)0)
      return -ENODEV;
   if (vfsmnt->mnt_quotas[type] != (struct file *)0)
      return -EBUSY;
   if ((error = getname(path, &tmp)) != 0)
      return (error);
   error = open_namei(tmp, O_RDWR, 0600, &inode, 0);
   putname(tmp);
   if (error)
      return (error);
   if (!S_ISREG(inode->i_mode)) {
      iput(inode);
      return -EACCES;
   }
   if (!inode->i_op || !inode->i_op->default_file_ops)
      goto end_quotaon;
   if ((filp = get_empty_filp()) == (struct file *)0)
      goto end_quotaon;
   filp->f_mode = (O_RDWR + 1) & O_ACCMODE;
   filp->f_flags = O_RDWR;
   filp->f_inode = inode;
   filp->f_pos = 0;
   filp->f_reada = 0;
   filp->f_op = inode->i_op->default_file_ops;
   if (filp->f_op->open)
      if (filp->f_op->open(filp->f_inode, filp))
         goto end_quotaon;
   if (!filp->f_op->read || !filp->f_op->write)
      goto end_quotaon;
   vfsmnt->mnt_quotas[type] = filp;
   vfs_open_filp(filp);
   dquot = dqget(dev, 0, type);
   vfsmnt->mnt_iexp[type] = (dquot) ? dquot->dq_itime : MAX_IQ_TIME;
   vfsmnt->mnt_bexp[type] = (dquot) ? dquot->dq_btime : MAX_DQ_TIME;
   dqput(dquot);
   add_dquot_ref(dev, type);
   return 0;
end_quotaon:
   if (filp != (struct file *)0)
      filp->f_count--;
   iput(inode);
   return -EIO;
}

/*
 * Just like iput, decrement referencecount of dquot.
 */
void dqput(struct dquot *dquot)
{
   if (!dquot)
      return;
   /*
    * If the dq_mnt pointer isn't initialized this entry needs no
    * checking and doesn't need to be written. It just an empty
    * dquot that is put back into the freelist.
    */
   if (dquot->dq_mnt != (struct vfsmount *)0) {
#if DEBUG_QUOTA > 0
      printk("VFS: dqput for %s %d, on dev (%d,%d)\n",
             quotatypes[dquot->dq_type], dquot->dq_id,
             MAJOR(dquot->dq_dev), MINOR(dquot->dq_dev));
      printk("VFS: New count will be %d\n", dquot->dq_count - 1);
#endif
      wait_on_dquot(dquot);
      if (!dquot->dq_count) {
         printk("VFS: iput: trying to free free dquot\n");
         printk("VFS: device %d/%d, dquot of %s %d\n",
                MAJOR(dquot->dq_dev), MINOR(dquot->dq_dev),
                quotatypes[dquot->dq_type], dquot->dq_id);
         return;
      }
repeat:
      if (dquot->dq_count > 1) {
         dquot->dq_count--;
         return;
      }
      wake_up(&dquot_wait);
      if (dquot->dq_flags & DQ_MOD) {
         write_dquot(dquot);   /* we can sleep - so do again */
         wait_on_dquot(dquot);
         goto repeat;
      }
   }
   if (dquot->dq_count) {
      dquot->dq_count--;
      nr_free_dquots++;
   }
   return;
}

static struct dquot *get_empty_dquot(void)
{
   struct dquot *dquot, *best;
   int i;

   if (nr_dquots < NR_INODE && nr_free_dquots < (nr_dquots >> 2))
      grow_dquots();

repeat:
   dquot = first_dquot;
   best = NODQUOT;
   for (i = 0; i < nr_dquots; dquot = dquot->dq_next, i++) {
      if (!dquot->dq_count) {
         if (!best)
            best = dquot;
         if (!(dquot->dq_flags & DQ_MOD) && !(dquot->dq_flags & DQ_LOCKED)) {
            best = dquot;
            break;
         }
      }
   }
   if (!best || best->dq_flags & DQ_MOD || best->dq_flags & DQ_LOCKED)
      if (nr_dquots < NR_DQUOTS) {
         grow_dquots();
         goto repeat;
      }
   dquot = best;
   if (!dquot) {
      printk("VFS: No free dquots - contact mvw@mcs.nl.mugnet.org\n");
      sleep_on(&dquot_wait);
      goto repeat;
   }
   if (dquot->dq_flags & DQ_LOCKED) {
      wait_on_dquot(dquot);
      goto repeat;
   }
   if (dquot->dq_flags & DQ_MOD) {
      write_dquot(dquot);
      goto repeat;
   }
   if (dquot->dq_count)
      goto repeat;
   clear_dquot(dquot);
   dquot->dq_count = 1;
   nr_free_dquots--;
   if (nr_free_dquots < 0) {
      printk ("VFS: get_empty_dquot: bad free dquot count.\n");
      nr_free_dquots = 0;
   }
   return dquot;
}

/*
 * Just like iget, increment referencecount of a dquot and return
 * pointer to it in the hashqueueu.
 */
struct dquot *dqget(dev_t dev, unsigned int id, short type)
{
   struct dquot *dquot, *empty;
   struct vfsmount *vfsmnt;

   if ((vfsmnt = lookup_vfsmnt(dev)) == (struct vfsmount *)0 ||
       (vfsmnt->mnt_quotas[type] == (struct file *)0))
      return (NODQUOT);
#if DEBUG_QUOTA > 0
   printk("VFS: dqget for %s %d, on dev (%d,%d)\n",
          quotatypes[type], id, MAJOR(dev), MINOR(dev));
#endif
   empty = get_empty_dquot();
repeat:
   dquot = *(hash(dev, id, type));
   while (dquot) {
      if (dquot->dq_dev != dev || dquot->dq_id != id) {
         dquot = dquot->dq_hash_next;
         continue;
      }
      wait_on_dquot(dquot);
      if (dquot->dq_dev != dev || dquot->dq_id != id)
         goto repeat;
      if (!dquot->dq_count)
         nr_free_dquots--;
      dquot->dq_count++;
      if (empty)
         dqput(empty);
      return dquot;
   }
   if (!empty)
      return (NODQUOT);
   dquot = empty;
   dquot->dq_id = id;
   dquot->dq_type = type;
   dquot->dq_dev = dev;
   dquot->dq_mnt = vfsmnt;
   put_last_free(dquot);
   insert_dquot_hash(dquot);
   read_dquot(dquot);
   return dquot;
}

/*
 * Initialize pointer in a inode to the right dquots.
 * Be smart and increment count only if already valid pointervalue.
 */
void getinoquota(struct inode *inode, short type)
{
   unsigned int id = 0;
   short cnt;

   for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
      if (type != -1 && cnt != type)
         continue;
      if (inode->i_dquot[cnt] != NODQUOT) {
#if DEBUG_QUOTA > 0
         printk("VFS: Increment already initialized dquot ptr for %s %d\n",
                quotatypes[cnt], inode->i_dquot[cnt]->dq_id);
#endif
         inode->i_dquot[cnt]->dq_count++;
         continue;
      }
      switch (cnt) {
         case USRQUOTA:
            id = inode->i_uid;
            break;
         case GRPQUOTA:
            id = inode->i_gid;
            break;
      }
      inode->i_dquot[cnt] = dqget(inode->i_dev, id, cnt);
   }
}

void putinoquota(struct inode *inode)
{
   short cnt;

   for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
      if (inode->i_dquot[cnt] == NODQUOT)
         continue;
      dqput(inode->i_dquot[cnt]);
      if (!inode->i_writecount)
         inode->i_dquot[cnt] = NODQUOT;
   }
}

/*
 * This are two simple algorithms that calculates the size of a file in blocks
 * and from a number of blocks to a isize.
 * It is not perfect but works most of the time.
 */
u_long isize_to_blocks(size_t isize, size_t blksize)
{
   u_long blocks;
   u_long indirect;

   if (!blksize)
      blksize = BLOCK_SIZE;
   blocks = (isize / blksize) + ((isize % blksize) ? 1 : 0);
   if (blocks > 10) {
      indirect = ((blocks - 11) >> 8) + 1; /* single indirect blocks */
      if (blocks > (10 + 256)) {
         indirect += ((blocks - 267) >> 16) + 1; /* double indirect blocks */
         if (blocks > (10 + 256 + (256 << 8)))
            indirect++; /* triple indirect blocks */
      }
      blocks += indirect;
   }
   return blocks;
}

size_t blocks_to_isize(u_long blocks, size_t blksize)
{
   size_t isize;
   u_long indirect;

   if (!blksize)
      blksize = BLOCK_SIZE;
   isize = blocks * blksize;
   if (blocks > 10) {
      indirect = ((blocks - 11) >> 8) + 1; /* single indirect blocks */
      if (blocks > (10 + 256)) {
         indirect += ((blocks - 267) >> 16) + 1; /* double indirect blocks */
         if (blocks > (10 + 256 + (256 << 8)))
            indirect++; /* triple indirect blocks */
      }
      isize -= indirect * blksize;
   }
   return isize;
}

/*
 * Allocate the number of inodes and blocks from a diskquota.
 */
int quota_alloc(struct inode *inode, u_long wanted_inodes, u_long wanted_blocks,
                u_long *avail_blocks)
{
   u_long availblocks = 0, local_avail;
   unsigned short cnt;

#if DEBUG_QUOTA > 1
   printk("VFS: quota_alloc for dev (%d,%d), uid %d, gid %d inodes %d, blocks %d\n",
          MAJOR(inode->i_dev), MINOR(inode->i_dev), inode->i_uid, inode->i_gid,
          wanted_inodes, wanted_blocks);
#endif
   availblocks = wanted_blocks;
   if (wanted_inodes > 0 || wanted_blocks > 0) {
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         if (inode->i_dquot[cnt] == NODQUOT)
            continue;
         if (check_idq(inode->i_dquot[cnt], inode->i_dquot[cnt]->dq_id,
                       cnt, wanted_inodes) == NO_QUOTA ||
             check_bdq(inode->i_dquot[cnt], inode->i_dquot[cnt]->dq_id,
                       cnt, wanted_blocks, &local_avail) == NO_QUOTA)
            return NO_QUOTA;
         availblocks = (availblocks == 0) ? local_avail : \
                       min(availblocks, local_avail);
      }
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         if (inode->i_dquot[cnt] == NODQUOT)
            continue;
         incr_quota(inode->i_dquot[cnt], wanted_inodes, availblocks);
      }
   }
   if (avail_blocks != (u_long *)0)
      *avail_blocks = availblocks;
   return QUOTA_OK;
   /* NOTREACHED */
}

/*
 * Remove the number of inodes and blocks from a diskquota.
 */
void quota_remove(struct inode *inode, u_long inodes, u_long blocks)
{
   short cnt;

#if DEBUG_QUOTA > 1
   printk("VFS: quota_remove for dev (%d,%d), uid %d, gid %d, ino %d, blocks %d\n", 
          MAJOR(inode->i_dev), MINOR(inode->i_dev), inode->i_uid, inode->i_gid,
          inodes, blocks);
#endif
   if (inodes > 0 || blocks > 0) {
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         if (inode->i_dquot[cnt] == NODQUOT)
            continue;
         decr_quota(inode->i_dquot[cnt], inodes, blocks);
      }
   }
}

/*
 * Transfer the number of inode and blocks from one diskquota to an other.
 */
int quota_transfer(struct inode *inode, uid_t newuid, gid_t newgid,
                   u_long inodes, u_long blocks)
{
   struct dquot *transfer_to[MAXQUOTAS];
   u_long availblocks;
   unsigned int id = 0;
   short cnt, disc;

#if DEBUG_QUOTA > 1
   printk("VFS: quota_transfer for dev (%d,%d), uid %d to %d, gid %d to %d, ino %d, blocks %d\n", 
          MAJOR(inode->i_dev), MINOR(inode->i_dev), inode->i_uid, newuid,
          inode->i_gid, newgid, inodes, blocks);
#endif

   if (inodes > 0 || blocks > 0) {
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         transfer_to[cnt] = NODQUOT;
         switch(cnt) {
            case USRQUOTA:
               if (inode->i_uid == newuid)
                  continue;
               id = newuid;
               break;
            case GRPQUOTA:
               if (inode->i_gid == newgid)
                  continue;
               id = newgid;
               break;
         }
         transfer_to[cnt] = dqget(inode->i_dev, id, cnt);
         if (transfer_to[cnt] == NODQUOT)
            continue;

         if (check_idq(transfer_to[cnt], id, cnt, inodes) == NO_QUOTA ||
             check_bdq(transfer_to[cnt], id, cnt, blocks, &availblocks) == NO_QUOTA ||
             availblocks != blocks) {
            for (disc = 0; disc <= cnt; disc++)
               dqput(transfer_to[disc]);
            return NO_QUOTA;
         }
      }
      for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
         if (transfer_to[cnt] == NODQUOT)
            continue;
         decr_quota(inode->i_dquot[cnt], inodes, blocks);
         incr_quota(transfer_to[cnt], inodes, blocks);
         transfer_to[cnt]->dq_count += inode->i_writecount;
         inode->i_dquot[cnt]->dq_count -= inode->i_writecount;
         dqput(inode->i_dquot[cnt]);
         inode->i_dquot[cnt] = transfer_to[cnt];
      }
   }
   return QUOTA_OK;
   /* NOTREACHED */
}

void quota_init(void)
{
   memset(hash_table, 0, sizeof(hash_table));
   first_dquot = NODQUOT;
}

/*
 * Ok this is the systemcall interface, this communicates with
 * the userlevel programs.
 */
asmlinkage int sys_quotactl(int cmd, const char *special, int id, caddr_t addr)
{
   int cmds = 0, type = 0, flags = 0;
   struct vfsmount *vfsmnt;
   struct inode *ino;
   dev_t dev;

   cmds = cmd >> SUBCMDSHIFT;
   type = cmd & SUBCMDMASK;

   if ((u_int) type >= MAXQUOTAS)
      return -EINVAL;
   switch (cmds) {
      case Q_SYNC:
         break;
      case Q_GETQUOTA:
         if (((type == USRQUOTA && current->uid != id) ||
              (type == GRPQUOTA && current->gid != id)) && !suser())
            return -EPERM;
         break;
      default:
         if (!suser())
            return -EPERM;
   }

   if (special == (char *)0 && cmds == Q_SYNC)
      dev = 0;
   else {
      if (namei(special, &ino))
         return -EINVAL;
      dev = ino->i_rdev;
      if (!S_ISBLK(ino->i_mode)) {
         iput(ino);
         return -ENOTBLK;
      }
      iput(ino);
   }

   switch (cmds) {
      case Q_QUOTAON:
         return quota_on(dev, type, (char *) addr);
      case Q_QUOTAOFF:
         return quota_off(dev, type);
      case Q_GETQUOTA:
         return get_quota(dev, id, type, (struct dqblk *) addr);
      case Q_SETQUOTA:
         flags |= SET_QUOTA;
         break;
      case Q_SETUSE:
         flags |= SET_USE;
         break;
      case Q_SETQLIM:
         flags |= SET_QLIMIT;
         break;
      case Q_SYNC:
         return sync_dquots(dev, type);
      default:
         return -EINVAL;
   }

   flags |= QUOTA_SYSCALL;
   if ((vfsmnt = lookup_vfsmnt(dev)) != (struct vfsmount *)0 &&
        vfsmnt->mnt_quotas[type] != (struct file *)0)
      return set_dqblk(dev, id, type, flags, (struct dqblk *) addr);
   return -ESRCH;
   /* NOTREACHED */
}

#else

asmlinkage int sys_quotactl(int cmd, const char *special, int id, caddr_t addr)
{
   return -ENOSYS;
}

#endif /* CONFIG_QUOTA */

