/*
 *	simple_ls.C -- A simple directory listing program (that has no where near the power of ls)
 */

/*
 *	simple_ls demonstrates how to create a simple filemgr-like directory listing.
 *
 * Functions used:
 *
 */

/* Header Files */

# include <OI/oi.H>
# include <sys/types.h>
# include <sys/stat.h>
# include <string.h>
#if defined(ultrix) || defined(SVR4) || defined(hpux)
#include <unistd.h>
#else
#include <sysent.h>
#endif
# include <dirent.h>
# include <stdlib.h>
	
/* Macro Definitions */

/* Variable Definitions */

/* Function Declarations */

/*
 *	Icon
 *
 *	The icon class is a box that has a glyph and a static text label in it.  We use this as a base
 *	class for icons we really display.
 */
class Icon : public OI_box {
 public:
			Icon (OI_glyph *, char *);
			~Icon();
 protected:
	void		clone_glyph (OI_glyph *);
	OI_glyph	*create_icon (char *, char *, OI_number, OI_number, char *);
};
Icon::~Icon() { }

/*
 *	DirIcon
 *
 *	DirIcon is used to display directories.  It is derived from icon and adds a static directory icon that gets
 *	cloned for every directory that we have.
 */
class DirIcon : public Icon {
 public:
			DirIcon (char *);
			~DirIcon();
 private:
	static OI_glyph	*dir_icon;
};
DirIcon::~DirIcon() { }

OI_glyph* DirIcon::dir_icon;

/*
 *	ExecIcon
 *
 *	ExecIcon is used to display executable files.  It is derived from icon and adds a static directory icon that gets
 *	cloned for every executable that we have.
 */
class ExecIcon : public Icon {
 public:
			ExecIcon (char *);
			~ExecIcon();
 private:
	static OI_glyph	*exec_icon;
};
OI_glyph* ExecIcon::exec_icon;
ExecIcon::~ExecIcon() { }

/*
 *	AsciiIcon
 *
 *	AsciiIcon is used to display normal files.  It is derived from icon and adds a static directory icon that gets
 *	cloned for every normal file that we have.
 */
class AsciiIcon : public Icon {
 public:
			AsciiIcon (char *);
			~AsciiIcon();
 private:
	static OI_glyph	*ascii_icon;
};
AsciiIcon::~AsciiIcon() { }
OI_glyph* AsciiIcon::ascii_icon;

/*
 *	Icon::Icon
 *
 *	Constructor for Icon class.  Makes a box, puts an icon and label in it.
 */
Icon::Icon (OI_glyph *gp, char *lab) : OI_box ("Icon", 1, 1)
{
	OI_static_text	*stp;
	
	/*
	 *	Set frame width to 0 and make this layout.
	 */
	set_layout (OI_layout_row);
	set_frame_width (0);
	
	/*
	 *	If the glyph exist, clone it.
	 */
	if (gp)
		clone_glyph (gp);

	/*
	 *	Create a centered label.
	 */
	stp = oi_create_static_text ("label", lab);
	stp -> set_gravity (OI_grav_center);
	stp -> layout_associated_object (this, 2, 1, OI_ACTIVE);
}

/*
 *	Icon::clone_glyph
 *
 *	Clones a glyph and adds it to the layout.  This hack is due to the fact that I need to get the glyph
 *	initialized after Icon returns from the constructor the first time.  This lets the derived classes add the
 *	glyph later on.
 */
void Icon::clone_glyph (OI_glyph *gp)
{
	OI_glyph	*glyph;
	
	glyph = (OI_glyph *) gp -> clone();
	glyph -> layout_associated_object (this, 1, 1, OI_ACTIVE);
}

/*
 *	Icon::create_icon
 *
 *	Creates an icon of the specifed color, using the bits bm and bm_bg.  Both of these bitmaps must be the same
 *	size, or the glyph code hickups.
 */
OI_glyph *Icon::create_icon (char *bm, char *bm_bg, OI_number width, OI_number height, char *color)
{
	OI_glyph *gp;
	OI_pic_spec_mask *masks[2];
	PIXEL bg_pxl;
	
	/*
	 *	If the connection is color, allocate the named color.  Otherwise just use the background color.
	 */
	if (connection()->is_color())
		connection()->str_color (color, &bg_pxl);
	else
			bg_pxl = bkg_pixel();
		
	/*
	 *	Create the masks, then create the glyph.
	 */
	masks[0] = oi_create_pic_data_mask (bm, OI_PIC_FG, OI_PIC_FG, 0, 0);
	masks[1] = oi_create_pic_data_mask (bm_bg, OI_PIC_NONE, OI_PIC_NONE, bg_pxl, bg_pxl);
	gp = oi_create_glyph ("Icon", 2, masks, "", width, height, OI_NO, OI_NO);

	return gp;
}

/*
 *	DirIcon::DirIcon
 *
 *	Create a new directory icon.  If this is the first time, we need to really make the glyph.  Other times the
 *	Icon constructor will take care of it.
 */
DirIcon::DirIcon (char *dir) : Icon (dir_icon, dir)
{
#include "../bitmaps/dirc.xbm"
#include "../bitmaps/dirc-bg.xbm"
	if (!dir_icon) {
		dir_icon = create_icon (dirc_bits, dirc_bg_bits, dirc_width, dirc_height, "MediumPurple");
		clone_glyph (dir_icon);
	}
}

/*
 *	ExecIcon::ExecIcon
 *
 *	Create a new executable icon.  If this is the first time, we need to really make the glyph.  Other times the
 *	Icon constructor will take care of it.
 */
ExecIcon::ExecIcon (char *exec) : Icon (exec_icon, exec)
{
#include "../bitmaps/exec.xbm"
#include "../bitmaps/exec-bg.xbm"
	if (!exec_icon) {
		exec_icon = create_icon (exec_bits, exec_bg_bits, exec_width, exec_height, "Plum");
		clone_glyph (exec_icon);
	}
}

/*
 *	AsciiIcon::AsciiIcon
 *
 *	Create a new normal file icon.  If this is the first time, we need to really make the glyph.  Other times the
 *	Icon constructor will take care of it.
 */
AsciiIcon::AsciiIcon (char *ascii) : Icon (ascii_icon, ascii)
{
#include "../bitmaps/ascii.xbm"
#include "../bitmaps/ascii-bg.xbm"
	if (!ascii_icon) {
		ascii_icon = create_icon (ascii_bits, ascii_bg_bits, ascii_width, ascii_height, "PaleTurquoise");
		clone_glyph (ascii_icon);
	}
}

/*
 *	make_dir_box
 *
 *	Creates a box that has all the entries in the directory in it.  No sorting of the directory is done just yet,
 *	as that would obscure the OI techniques being demonstrated.
 */
OI_box *make_dir_box (char *dir)
{
	struct stat		sb;
	OI_box			*boxp;
	int			row;
	int			col;
	char			path[MAXPATHLEN];
	DIR			*dirp;
	struct dirent		*dirent;
	Icon			*icon;
	
	/*
	 *	Stat the directory to see if it is there.
	 */
	if (stat (dir, &sb)) {
		fprintf (stderr, "Can't stat directory %s\n", dir);
		exit (1);
	}

	/*
	 *	Check to see if it is a directory.  If not fail.
	 */
	if ((sb.st_mode & S_IFMT) != S_IFDIR) {
		fprintf (stderr, "%s is not a directory", dir);
		exit (1);
	}

	/*
	 *	Create a box to hold the directory.  A more advanced version would use scroll box here that size tracked.
	 *	A suspend layout is done to avoid O(n^2) layout behaviour.
	 */
	boxp = oi_create_box ("dirBox", 1, 1);
	boxp->set_layout (OI_layout_column);
	boxp->suspend_layout();
	
	/*
	 *	Open the directory.  While this shouldn't fail, I'll be paranoid just in case...
	 */
	if ((dirp = opendir (dir)) == NULL) {
		fprintf (stderr, "Logic says this can't happen, but I can't opendir %s\n", dir);
		exit (1);
	}

	/*
	 *	Read the directory in.  For each file, make a new icon specific to each type of file.  Then layout the new icon.
	 *	Be really dumb about placement of icons.  A more advanced program would sort them and arrange them into n
	 *	columns.
	 */
	row = 1;
	col = 1;
	readdir (dirp);		// Skip .
	readdir (dirp);		// Skip ..
	while (dirent = readdir (dirp)) {
		sprintf (path, "%s/%s", dir, dirent->d_name);
		if (stat (path, &sb))
			continue;
		if ((sb.st_mode & S_IFMT) == S_IFDIR)
			icon = new DirIcon (dirent->d_name);
		else if (sb.st_mode & S_IEXEC)
			icon = new ExecIcon (dirent->d_name);
		else
			icon = new AsciiIcon (dirent->d_name);
		icon->layout_associated_object (boxp, col, row, OI_ACTIVE);
		col++;
		if (col > 6) {
			col = 1;
			row++;
		}
	}
	closedir (dirp);

	/*
	 *	The resume layout here is supposed to be faster since we only layout each object once, not n times.
	 */
	boxp->resume_layout();
	return boxp;
}

/*
 *	Main
 *
 *	Main program.
 */
void
main(int argc, char *argv[])
{
	OI_connection	*conp;
	OI_app_window	*top_level;
	OI_box		*dir_box;
	
	/*
	 *	Open a connection.
	 */
	if (conp = OI_init (&argc, argv, "Simple Directory Listing program"))
	{
		/*
		 *	Create the OI_tests main window.  Make it row layout.
		 */
		top_level = oi_create_app_window("topLevel",1,1,"Simple 'ls'");
		top_level->set_layout (OI_layout_row);

		/*
		 *	Create a new directory object.
		 */
		dir_box = make_dir_box (argc > 1 ? argv[1] : ".");
		dir_box->layout_associated_object (top_level, 1, 1, OI_ACTIVE);
		
		/*
		 *	OK, display main window.
		 */
		top_level->set_associated_object(top_level->root(),OI_DEF_LOC, OI_DEF_LOC,OI_ACTIVE);
		OI_begin_interaction();
	}

	/*
	 * Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
}
