/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

/*
**	Main program and support routines for call programs.
**
**	Provides pattern matching for {pattern, action} pairs in state tables
**	setup by one of the call scripts in Callers/*.c (see PatList in caller.h).
*/

#ifndef	lint
static char	sccsid[]	= "@(#)caller.c	1.22 88/07/12";
#endif

#include	"global.h"
#include	"debug.h"

#include	"caller.h"

#include	<setjmp.h>
#include	<signal.h>



PatList *	cp;
PatList *	current;
char *		delim		= ":\n\r";	/* Delimiters for input matching */
jmp_buf		FlushJmp;
char		input[LINESIZE+1];
int		inputlen;			/* Length of "input" */
char *		Name;
char		Ret[10][LINESIZE+1];		/* For regex() results */
int		Traceflag;

extern char *	__loc1;				/* Start of match */

extern bool	match();
extern unsigned	alarm();



main(argc, argv)
	int	argc;
	char	*argv[];
{
	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	init(argc, argv);	/* Initialise call script */
	state(current);		/* Check initialisation */

	for ( ;; )
	{
		if ( cp->pattern == NULLSTR )
			reset();

		if ( !match(cp) )
			next();
	}
}



/*
**	Compile the pattern if first time round,
**	match the pattern, collecting any strings output,
**	then call the supplied function, with arguments specifying:
**		where the match started,
**		where it ended,
**		and addresses of matched strings.
*/

bool
match(pp)
	register PatList *	pp;
{
	register char *		result;

	if
	(
		pp->comp_pat == NULLSTR
		&&
		(pp->comp_pat = regcmp(pp->pattern, NULLSTR)) == NULLSTR
	)
	{
		Command("fail cannot compile pattern - \"", pp->pattern, "\"", NULLSTR);
		exit(1);
	}

	if ( Traceflag >= 3 )
	{
		out("trace match(\"");
		out(ExpandString(input, inputlen));
		out("\", \"");
		out(pp->pattern);
		out("\") ==> ");
	}

	if
	(
		(result = regex(pp->comp_pat, input,
				Ret[0], Ret[1], Ret[2], Ret[3], Ret[4],
				Ret[5], Ret[6], Ret[7], Ret[8], Ret[9]))
		!= NULLSTR
	)
	{
		if ( Traceflag >= 3 )
			outend("true");

		(void)(*pp->func)(__loc1, result,
				Ret[0], Ret[1], Ret[2], Ret[3], Ret[4],
				Ret[5], Ret[6], Ret[7], Ret[8], Ret[9]);

		return true;
	}

	if ( Traceflag >= 3 )
		outend("false");

	return false;
}



/*
**	Read in new input up to one of "delim".
*/

void
getinput()
{
	register int	i;

	for ( i = 0 ; i < LINESIZE ; )
	{
		char	c;

		if ( read(0, &c, 1) <= 0 )
		{
			strcpy(input, EOFSTR);
			inputlen = strlen(input);
			return;
		}

		if ( (c &= 0x7f) == 0 )
			continue;

		input[i++] = c;

		if ( strchr(delim, c) )
			break;
	}

	input[i] = '\0';
	inputlen = i;

	if ( i = Traceflag )
	{
		Traceflag = 0;
		Command("trace read ", input, NULLSTR);
		Traceflag = i;
	}
}



/*
**	output a string
*/

void
out(s)
	char *	s;
{
	(void)write(1, s, strlen(s));
}



/*
**	output a string terminated with a '\0'
*/

void
outend(s)
	char *	s;
{
	(void)write(1, s, strlen(s)+1);
}



/*
**	Combine above for NULLSTR terminated list.
*/

void
Command(s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12)
	char *		s1;
	char *		s2;
	char *		s3;
	char *		s4;
	char *		s5;
	char *		s6;
	char *		s7;
	char *		s8;
	char *		s9;
	char *		s10;
	char *		s11;
	char *		s12;
{
	char **		ap;
	register char *	cp;
	register int	n;

	if ( (n = Traceflag) > 0 && strncmp(s1, "trace ", 6) != STREQUAL )
	{
		Traceflag = 0;
		Command("trace ", s1, s2, s3, s4, s5, s6, s7, NULLSTR);
		Traceflag = n;
	}

	for ( ap = &s1 ; (cp = ARGS_OFF_STACK(ap)) != NULLSTR ; )
		(void)write(1, cp, strlen(cp));

	(void)write(1, "", 1);
}



void
state(p)
	register PatList *	p;
{
	if ( p == (PatList *)0 || p->pattern == NULLSTR )
	{
		Command("fail null pattern list", NULLSTR);
		exit(1);
	}

	current = p;
	cp = p;
}



next()
{
	cp++;	/* move to next pattern */
}



void
reset()
{
	cp = current;
	if ( Traceflag >= 3 )
		outend("trace reset");
	getinput();
}



/*
**	Extract arguments from passed file,
**	and pass them back via the passed function
**	to look like program invoked arguments.
*/

bool
readargs(file, funcp)
	char *		file;
	int		(*funcp)();
{
	register char *	ap;
	register int	i;
	VarArgs		va;

	if ( i = Traceflag )
	{
		Traceflag = 0;
		Command("trace readargs(", file, ")", NULLSTR);
		Traceflag = i;
	}

	if ( (ap = ReadFile(file)) == NULLSTR )
		return false;

	FIRSTARG(&va) = Name;

	if ( SplitArg(&va, ap) >= MAXVARARGS )	/* SplitArg processes quotes */
		Command("fail Too many arguments in \"", file, "\"", NULLSTR);

	(void)(*funcp)(NARGS(&va), &ARG(&va, 0));
	free(ap);

	return true;
}



/*
**	Clean up after Error()
*/

void
finish(err)
	int	err;
{
	char	numb[12];

	(void)sprintf(numb, "%d", err);
	Command("fail error type ", numb, NULLSTR);

	exit(1);
}



flushcatch(sig)
	int	sig;
{
	(void)signal(sig, SIG_IGN);
	longjmp(FlushJmp, 1);
}



void
flushinput()
{
	cp = current;

	if ( setjmp(FlushJmp) )
	{
		if ( Traceflag >= 2 )
			outend("trace flushinput returns");
		return;
	}
	
	(void)signal(SIGALRM, flushcatch);

	for(;;)
	{
		char		c[65];
		register int	n;

		(void)alarm(MINSLEEP + 1);
		n = read(0, c, sizeof c - 1);
		(void)alarm(0);
		if ( n <= 0 )
			break;
		if ( Traceflag >= 2 )
		{
			register char *	p;

			out("trace flushed \"");
			c[n] = '\0';
			out(p = expandstr(c));
			free(p);
			outend("\"");
		}
	}

	outend("trace flushinput read eof/err");
}



/*
**	Superceded quoting (now done in SplitArg).
*/

char *
expandstr(str)
	register char *	str;
{
	register char *	p;
	char *		res;
	static char	expanders[] = "befnrst\\";
	static char	expandees[] = "\b\\\f\n\r \t\\";

	res = p = Malloc(strlen(str) + 1);

	for ( ; *str ; p++, str++ )
	{
		if ( *str == '\\' && str[1] != '\0' )
		{
			register char *	q;

			if ( (q = strchr(expanders, *++str)) == NULLSTR )
			{
				*p++ = '\\';
				*p = *str;
			}
			else
				*p = expandees[q - expanders];
		}
		else
			*p = *str;
	}
	*p = '\0';

	return res;
}
