/*
**	patterns and functions to call a host via CSIRONET
**	using dmtmelb with "con" as a relay
**
**	SCCSID %W% %E%
*/

#define	MAXDEV	8

#include	"global.h"
#include	"debug.h"

#include	"caller.h"

char
	*CallArgs= CALLARGS,		/* optional args for call */
	*Daemon1 = NNDAEMON,		/* standard daemon */
	*Daemon2 = NN2DAEMON,		/* alternate protocol */
	*HostFlg = "-BCF";		/* cooked protocol over csironet */

char
	*Retry	 = "30",		/* seconds between open attempts */
	*Retries = "3",			/* times to attempt open */
	*Delay	 = "15";		/* TIMEOUT every delay seconds */

int
	Wait	 = 4;			/* mean minutes to wait after failure */

char
	*Host,				/* Who am i */
	*Target,			/* Target host */
	*CsiroReq,			/* Request to set-up virtual circuit */
	*Passwd,			/* Password at Target */
	*Device[MAXDEV],		/* Device(s) for link */
	*Speed;				/* Speed of link */

int
	Fcount	 = 0,	/* count of gross failures */
	Tcount	 = 0,	/* count of timeouts */
	Ccount	 = 0,	/* count of CSIRONET login attempts */
	Mcount	 = 0,	/* count of stray CSIRONET messages */
	Zcount	 = 0,	/* count of "csironet broken" type messages */
	Wcount	 = 0,	/* count of attempts to connect to dmtmelb */
	Lcount	 = 0;	/* count of Unix login attempts */

int
	MaxH	 = 6,
	MaxF	 = 3,
	MaxT	 = 4,
	MaxC	 = 3,
	MaxM	 = 6,
	MaxZ	 = 3,
	MaxW	 = 3,
	MaxL	 = 3;

char	*pump;		/* current string to prompt response from remote */
PatList *tostate;	/* state to enter on timeout */
char	*CurState = "?";/* string form of current state (for messages) */
char	*StateMsg = "?";/* string form to return to after timeout */

int	devices = 0;

int
	AtCSIRONET(),
	GotEof(),
	GotLogin(),
	GotPasswd(),
	GotTimeout(),
	CSIROlogin(),
	CSIRONETbusted(),
	Failure(),
	NetLogin(),
	SlowHost(),
	NoShellFail(),
	Start1(),
	Start2(),
	ToCSIRONET(),
	ToTarget(),
	reset();


/*
**	CSIRONET messages
*/

char	Disconn[]	= "DISCONNECT SEEN FROM";
char	Invalid[]	= "INVALID LOGIN REQUEST";
char	Noresp[]	= "NO RESPONSE FROM UNIX";
char	Unassi[]	= "TERMINAL IS UNASSIGNED";
char	Unixserv[]	= "UNIX SERVICE";
char	Unrec[]		= "UNRECOGNISED FIRST RECORD";
char	Notcomm[]	= "NOT COMMUNICATING";
char	Nofree[]	= "NO FREE PORTS";
char	Noport[]	= "NO PORT AVAILABLE";

/*
**	Other common patterns
*/

char	Eof[]		= EOFSTR;
char	Hung[]		= "[Ll]ine hung";
char	Login[]		= "[Ll]ogin";
char	LoginI[]	= "[Ll]ogin +[Ii]ncorrect";
char	Password[]	= "^[Pp]assword";
char	WPassword[]	= "[Ww]rong +[Pp]assword";
char	Timeout[]	= TIMEOUT;

/*
**	The pattern/action sets
**
**		normal "good" flow is to match the first pattern in each list
*/

PatList dmtlogin[] =
{
	{ Login, NetLogin },
	{ Timeout, Failure },
	{ Eof, GotEof },
	{0, 0}
};

PatList unixnet[] =
{
	{ "dmtmelb.*->", ToCSIRONET },
	{ Unrec, ToCSIRONET },
	{ Unassi, ToCSIRONET },
	{ Invalid, ToCSIRONET },
	{ Disconn, ToCSIRONET },
	{ Notcomm, CSIRONETbusted },
	{ Hung, Failure },
	{ "[Nn]o free port", Failure },		/* ?? is this right msg ?? */
	{ Timeout, Failure },
	{ Eof, GotEof },
	{0, 0}
};

PatList	CSIROnet[] =
{
	{ Unassi, CSIROlogin },
	{ Unrec, AtCSIRONET },
	{ Invalid, AtCSIRONET },
	{ Disconn, AtCSIRONET },
	{ Notcomm, CSIRONETbusted },
	{ Hung, Failure },
	{ Timeout, GotTimeout },
	{ Eof, GotEof },
	{ 0, 0}
};

PatList	CSIROreq[] =
{
	{ Unixserv, ToTarget },
	{ Nofree, Failure },
	{ Noport, Failure },
	{ Notcomm, CSIRONETbusted },
	{ Timeout, GotTimeout },
	{ Eof, GotEof },
	{ 0, 0}
};

PatList AtTarget[] =
{
	{ Login, GotLogin },
	{ Unrec, ToCSIRONET },
	{ Unassi, ToCSIRONET },
	{ Invalid, ToCSIRONET },
	{ Disconn, ToCSIRONET },
	{ Noresp, ToCSIRONET },
	{ Notcomm, CSIRONETbusted },
	{ Timeout, GotTimeout },
	{ Eof, GotEof },
	{ 0, 0 }
};

PatList TargetLogin[] =
{
	{ Password, GotPasswd },
	{ STARTMSG, Start1 },	/* some moron might have deleted passwd */
	{ START2MSG, Start2 },	/* alternate protocol */
	{ LoginI, ToTarget },
	{ "no shell", NoShellFail },
	{ Unrec, ToCSIRONET },
	{ Unassi, ToCSIRONET },
	{ Disconn, ToCSIRONET },
	{ Noresp, ToCSIRONET },
	{ Notcomm, CSIRONETbusted },
	{ Timeout, GotTimeout },
	{ Eof, GotEof },
	{ 0, 0 }
};

PatList DaemonWait[] =
{
	{ STARTMSG, Start1 },
	{ START2MSG, Start2 },			/* alternate protocol */
	{ LoginI, ToTarget },
	{ WPassword, ToTarget },		/* cretins ! */
	{ "no shell", NoShellFail },
	{ Unrec, ToCSIRONET },
	{ Unassi, ToCSIRONET },
	{ Disconn, ToCSIRONET },
	{ Noresp, ToCSIRONET },
	{ Notcomm, CSIRONETbusted },
	{ Timeout, SlowHost },
	{ Eof, GotEof },
	{ 0, 0 }
};

#define	State(s)	do { CurState = "s"; state(s); } while (0)

/*
**	Function to process arguments, either from NNcall, or from CALLARGS.
*/

void
Args(argc, argv)
	register int	argc;
	register char *	argv[];
{
	while ( --argc > 0 )
	{
		Trace2(1, "Args: \"%s\"", argv[1]);

		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case '1':
					Daemon1 = ++*argv;
					goto break2;

				case '2':
					Daemon2 = ++*argv;
					goto break2;

				case 'M':
					switch ( c = *++*argv )
					{
					case 'c':
						MaxC = atoi(++*argv);
						break;

					case 'f':
						MaxF = atoi(++*argv);
						break;

					case 'h':
						MaxH = atoi(++*argv);
						break;

					case 'l':
						MaxL = atoi(++*argv);
						break;

					case 'm':
						MaxM = atoi(++*argv);
						break;

					case 't':
						MaxT = atoi(++*argv);
						break;

					case 'w':
						MaxW = atoi(++*argv);
						break;

					case 'z':
						MaxZ = atoi(++*argv);
						break;
					}
					goto break2;

				case 'R':
					Retries = ++*argv;
					goto break2;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) == 0 )
						Traceflag = 1;
					goto break2;

				case 'd':
					if (devices < MAXDEV)
						Device[devices++] = ++*argv;
					goto break2;

				case 'l':
					Host = ++*argv;
					goto break2;

				case 'p':
					Passwd = expandstr(++*argv);
					goto break2;

				case 'r':
					Retry = ++*argv;
					goto break2;

				case 's':
					Speed = ++*argv;
					goto break2;

				case 't':
					Delay = ++*argv;
					goto break2;

				case 'v':
					CsiroReq = expandstr(++*argv);
					goto break2;

				case 'w':
					Wait = atoi(++*argv);
					goto break2;

				default:
					Command
					(
						"fail unexpected flag \"-",
						*argv,
						"\" in ",
						CallArgs,
						NULLSTR
					);
					exit(1);
				}
			}
break2:			;
		}
		else
			Target = *argv;
	}
}



init(argc, argv)
	int		argc;
	char *		argv[];
{
	register char *	cp;
	register i;

	Host = NodeName();

	Args(argc, argv);

	if ( (cp = strrchr(*argv, '/')) != NULLSTR )
	{
		*cp = '\0';
		cp = *argv;
	}
	else
		cp = ".";

	CallArgs = concat(cp, "/", CallArgs, NULLSTR);
	(void)readargs(CallArgs, Args);

	if (Target == NULLSTR || Passwd == NULLSTR || CsiroReq == NULLSTR ||
	    devices == 0 || Device[0] == NULLSTR)
	{
		outend("fail incomplete initialisation");
		exit(1);
	}

	srand((getpid() << 3) + (int)time((long *)0));

	Command("mode ", HostFlg, NULLSTR);
	Command("timeout ", Delay, NULLSTR);

	opndevice();

	if ( Speed != NULLSTR )
		Command("speed ", Speed, NULLSTR);

	Command("read", NULLSTR);

	toDMTmelb();
}

opndevice()
{
	register i;
	register char *dev;

	Command("retry ", Retry, " ", Retries, NULLSTR);

#ifdef UUCPLOCKS
	for (i = 0; i < devices; i++)
	{
		dev = strrchr(Device[i], '/');
		if (dev == NULLSTR)
			dev = Device[i];
		else
			dev++;
		if (mlock(dev) == 0)
			break;
	}
	if (i == devices)
	{
		if (devices == 1)
			Command("fail ", Device[0], " busy", NULLSTR);
		else
			outend("fail no available device");
		exit(1);
	}
	/*
	 * race here - someone else may grab this device before its
	 * locked again by NNcall as part of the "open" command.
	 * No harm done - we just miss out on this call
	 */
	rmlock(NULLSTR);
#else
	i = 0;
#endif

	Command("open ", Device[i], NULLSTR);
}

/*
 * Action functions - a normal connection sequence
 * uses each of the following functions, in order.
 */
toDMTmelb()
{
	Tcount = 0;
	Wcount = 0;
	outend("write @\r");
	State(dmtlogin);
	getinput();
}

NetLogin()
{
	if (++Wcount >= MaxW) {
		outend("fail unable to login at dmtmelb");
		exit(1);
	}
	Command("write :csiro\r", NULLSTR);
	State(unixnet);
	getinput();
}

ToCSIRONET()
{
	if (++Ccount >= MaxC) {
		if (Lcount > 0)
			Command("fail ", Target, " not answering", NULLSTR);
		else
			Command("fail can't connect through CSIRONET", NULLSTR);
		exit(1);
	}
	Tcount = 0;
	Mcount = 0;
	pump = "T\4";
	Command("sleep 2", NULLSTR);
	Command("write T\004", NULLSTR);
	State(CSIROnet);
	tostate = CSIROnet;
	StateMsg = CurState;
	getinput();
}

CSIROlogin()
{
	Mcount = 0;
	Command("write ", CsiroReq, "\r", NULLSTR);
	State(CSIROreq);
	getinput();
}

ToTarget()
{
	if (++Lcount >= MaxL) {
		Command("fail unable to login at ", Target, NULLSTR);
		exit(1);
	}
	outend("write \r");
	state(AtTarget);
	CurState = "TargetLogin";
	getinput();
}

GotLogin()
{
	Command("write ", Host, "\r", NULLSTR);
	pump = "\r";
	tostate = AtTarget;
	State(TargetLogin);
	StateMsg = CurState;
	getinput();
}

GotPasswd()
{
	Command("write ", Passwd, "\r", NULLSTR);
	State(DaemonWait);
	getinput();
}

Start1()
{
	Command("daemon ", Daemon1, NULLSTR);
	Command("succeed ", HostFlg, " ", Target, NULLSTR);
	exit(0);
}

/*
 * Daemon 2 is an alternative that the target host might select
 */
Start2()
{
	Command("daemon ", Daemon2, NULLSTR);
	Command("succeed ", HostFlg, " ", Target, NULLSTR);
	exit(0);
}

/*
 * The remaining functions handle various error
 * and unusual conditions that may be encountered
 */
CSIRONETbusted()
{
	if (++Zcount >= MaxZ) {
		outend("fail CSIRONET down");
		exit(1);
	}
	ToCSIRONET();
}

AtCSIRONET()
{
	Tcount = 0;
	if (++Mcount >= MaxM) {
		outend("fail CSIRONET misbehaving");
		exit(1);
	}
	outend("write T\04");
	getinput();
}

GotTimeout()
{
	if (++Tcount >= MaxT)
	{
		Command("fail timeout at ", CurState, NULLSTR);
		exit(1);
	}

	Command("write ", pump, NULLSTR);
	CurState = StateMsg;
	state(tostate);
}

Failure()
{
	register wait;

	Ccount = 0;
	if (++Fcount >= MaxF) {
		Command("fail repeated failures at ", CurState, NULLSTR);
		exit(1);
	}
	outend("close");

	wait = Wait * 60;
	wait += ((rand() >> 3) % (4 * 60)) - 2 * 60;
	if (wait < 20)
		wait = 20;
	sleep(wait);

	opndevice();

	if ( Speed != NULLSTR )
		Command("speed ", Speed, NULLSTR);

	Command("read", NULLSTR);
	toDMTmelb();
}

GotEof()
{
	outend("fail unexpected EOF");
	exit(1);
}

SlowHost()
{
	if (++Tcount >= MaxH) {
		Command("fail ", Target, " not responding", NULLSTR);
		exit(1);
	}
	reset();
}

NoShellFail()
{
	/*
	 * this is unrecoverable
	 */
	Command("fail no shell for ", Host, " at ", Target, NULLSTR);
	exit(1);
}

#ifdef UUCPLOCKS
cleanup(n)
{
	rmlock(NULLSTR);
	exit(n);
}
#endif
