/*
 * set / display tty modes
 *
 *	for MELB_TTY only
 */

#include <stdio.h>
#include <ctype.h>
#include <sgtty.h>
	/* redefinitions to make sgtty.h look more regular */
#define	sg_intr	t_intrc
#define	sg_quit	t_quitc
#define	sg_start t_startc
#define	sg_stop	t_stopc
#define	sg_eof	t_eofc
#define	sg_brkc	t_brkc
char *strcpy();

#ifndef	MELB_TTY
		;; ?? WRONG STTY ?? ;;
#endif

#define	WIDTH		72		/* maximum print width */
#define	CTLFMT		"'^%c'"		/* format to print control chars */
#define	pchar(c)	(c|'@')		/* printable form of ctl char */
#define	CHFMT		"'%c'"		/* format for regular characters */
#define	OCTFMT		"'\\%.1o'"	/* format for char in octal */

#define	HZ		60		/* clock rate */
#define	ms(n)		(((n*HZ)+999)/1000)
#define	equ(a,b)	(strcmp(a,b)==0)

struct sgttyb mode;
struct tchars mod;
struct sgttychr chr;
struct sgttydel del;
struct sgttyst state;
struct sgttyctl cntl;
struct sgttypag pag;

struct sgttyb defmode ={
	0,	0,	ctl('h'), '@',	ECHO|CRMOD|ANYP|KNL
};

struct tchars defmod ={
	0177, ctl('\\'), ctl('q'), ctl('s'), ctl('d'), 0200
};

struct sgttychr defchr ={
	ctl('a'), 033, ctl('o'), ctl('r'), ctl('x'), 0177
};

struct sgttydel defdel ={
	0,	0,	XTABS,	0,	0,	0
};

struct sgttyst defstate ={
	0,	UNKNOWN,	0,	0
};

struct sgttyctl defcntl ={
	0xf7ffc97f,	'\\',	'#'
};

struct sgttypag defpag ={
	0,	0,	0,	'\200',	'\200'
};

struct {
	char *string;

	int	set;
	int	reset;
} modes [] ={
	"even",		EVENP,		0,
	"-even",	0,		EVENP,
	"odd",		ODDP,		0,
	"-odd",		0,		ODDP,
	"raw",		RAW,		0,
	"-raw",		0,		RAW,
	"cbreak",	CBREAK,		0,
	"-cbreak",	0,		CBREAK,
	"cooked",	0,		RAW|CBREAK,
	"-nl",		CRMOD,		0,
	"nl",		0,		CRMOD,
	"echo",		ECHO,		0,
	"-echo",	0,		ECHO,
	"lcase",	LCASE,		0,
	"-lcase",	0,		LCASE,
	"ucase",	UCASE,		0,
	"-ucase",	0,		UCASE,
	"dcase",	DCASE,		0,
	"-dcase",	0,		DCASE,
	"knl",		KNL,		0,
	"-knl",		0,		KNL,
	"vdu",		RO_VDU,		RO_TYPE,
	"-vdu",		0,		RO_TYPE,
	"ero",		RO_HARD,	RO_TYPE,
	"-ero",		0,		RO_TYPE,
	"wrap",		WRAP,		0,
	"-wrap",	0,		WRAP,
	"page",		PAGE,		0,
	"-page",	0,		PAGE,
	"hold",		HOLDSTOP,	0,
	"-hold",	0,		HOLDSTOP,
	0
};

struct {
	char *string;
	short pgset;
	short pgreset;
} pgflags [] ={
	"halt",		PG_STOP,	0,
	"-halt",	0,		PG_STOP,
	"pause",	PG_PAUS,	0,
	"-pause",	0,		PG_PAUS,
	"fixed",	PG_FIXD,	0,
	"-fixed",	0,		PG_FIXD,
	"clear",	PG_CLEAR,	0,
	"-clear",	0,		PG_CLEAR,
	"nlgo",		PG_NLGO,	0,
	"-nlgo",	0,		PG_NLGO,
	0
};

struct {
	char *string;
	char *chrp;
	char	defprt;
} chars[] ={
	"dchar",	&del.sg_dchar,		0,
	"erase",	&mode.sg_erase,		0,
	"kill",		&mode.sg_kill,		0,
	"start",	&mod.sg_start,		ctl('q'),
	"stop",		&mod.sg_stop,		ctl('s'),
	"intr",		&mod.sg_intr,		0,
	"quit",		&mod.sg_quit,		ctl('\\'),
	"eof",		&mod.sg_eof,		ctl('d'),
	"redisp",	&chr.sg_redisp,		ctl('a'),
	"escp",		&chr.sg_escp,		033,
	"disc",		&chr.sg_disc,		ctl('o'),
	"dispq",	&chr.sg_dqueue,		ctl('r'),
	"delq",		&chr.sg_delq,		ctl('x'),
	"break",	&chr.sg_brkin,		0177,
	"desc",		&cntl.sg_desc,		'\\',
	"ddel",		&cntl.sg_delet,		'#',
	"brk",		&mod.sg_brkc,		0200,
	"pgon",		&pag.sg_pgon,		0200,
	"pgoff",	&pag.sg_pgoff,		0200,
	0
};

struct {
	char *string;
	char *chrp;
	char  ccval;
} cval[] ={
	"ek",		&mode.sg_erase,		ctl('h'),
	"ek",		&mode.sg_kill,		'@',
	"tek4012",	&del.sg_ffdel,		ms(800),
	"tek4012",	&state.sg_width,	72,
	"tek4012",	&state.sg_depth,	35,	/** ?? **/
	"int8001",	&del.sg_ffdel,		ms(50),
	"int8001",	&del.sg_nldel,		ms(50),
	"int8001",	&state.sg_width,	80,
	"int8001",	&state.sg_depth,	48,
	"vc404",	&state.sg_width,	80,
	"vc404",	&state.sg_depth,	24,
	"vc404",	&del.sg_ffdel,		ms(40),
	"try1061",	&state.sg_width,	80,
	"try1061",	&state.sg_depth,	24,
	"cdc713",	&state.sg_width,	80,
	"cdc713",	&state.sg_depth,	13,	/** ?? **/
	"ti700",	&state.sg_width,	80,
	"ti700",	&state.sg_depth,	0,
	"tty43",	&state.sg_width,	132,
	"tty43",	&state.sg_depth,	48,	/** ?? **/
	"la36",		&state.sg_width,	132,
	"la36",		&state.sg_depth,	66,
	"car300",	&mod.sg_stop,		ctl('t'),
	"car300",	&mod.sg_start,		ctl('r'),
	"car300",	&state.sg_width,	132,
	"car300",	&state.sg_depth,	66,
	"diab1620",	&state.sg_width,	132,
	"diab1620",	&state.sg_depth,	66,
	"adm3a",	&state.sg_width,	80,
	"adm3a",	&state.sg_depth,	24,
	"hzl1500",	&state.sg_width,	80,
	"hzl1500",	&state.sg_depth,	24,
	"vis200",	&state.sg_width,	80,
	"vis200",	&state.sg_depth,	24,
	"ads520",	&state.sg_width,	80,
	"ads520",	&state.sg_depth,	24,
	"tabs",		&del.sg_tabdel,		0,
	"-tabs",	&del.sg_tabdel,		XTABS,
	"form",		&del.sg_ffdel,		0,
	"-form",	&del.sg_ffdel,		XFORM,
	"ff0",		&del.sg_ffdel,		0,
	"ff1",		&del.sg_ffdel,		0177,
	"nl0",		&del.sg_nldel,		0,
	"nl1",		&del.sg_nldel,		NL_37,
	"nl2",		&del.sg_nldel,		6,
	"nl3",		&del.sg_nldel,		0,
	"cr0",		&del.sg_crdel,		0,
	"cr1",		&del.sg_crdel,		5,
	"cr2",		&del.sg_crdel,		10,
	"cr3",		&del.sg_crdel,		0,
	"tab0",		&del.sg_tabdel,		0,
	"tab1",		&del.sg_tabdel,		TAB_37,
	"tab2",		&del.sg_tabdel,		0,
	"tab3",		&del.sg_tabdel,		0,
	"9600",		&mode.sg_ispeed,	B9600,
	"9600",		&mode.sg_ospeed,	B9600,
	"4800",		&mode.sg_ispeed,	B4800,
	"4800",		&mode.sg_ospeed,	B4800,
	"2400",		&mode.sg_ispeed,	B2400,
	"2400",		&mode.sg_ospeed,	B2400,
	"1800",		&mode.sg_ispeed,	B1800,
	"1800",		&mode.sg_ospeed,	B1800,
	"1200",		&mode.sg_ispeed,	B1200,
	"1200",		&mode.sg_ospeed,	B1200,
	"600",		&mode.sg_ispeed,	B600,
	"600",		&mode.sg_ospeed,	B600,
	"300",		&mode.sg_ispeed,	B300,
	"300",		&mode.sg_ospeed,	B300,
	"200",		&mode.sg_ispeed,	B200,
	"200",		&mode.sg_ospeed,	B200,
	"150",		&mode.sg_ispeed,	B150,
	"150",		&mode.sg_ospeed,	B150,
	"134",		&mode.sg_ispeed,	B134,
	"134",		&mode.sg_ospeed,	B134,
	"110",		&mode.sg_ispeed,	B110,
	"110",		&mode.sg_ospeed,	B110,
	"75",		&mode.sg_ispeed,	B75,
	"75",		&mode.sg_ospeed,	B75,
	"50",		&mode.sg_ispeed,	B50,
	"50",		&mode.sg_ospeed,	B50,
	"0",		&mode.sg_ispeed,	B0,
	"0",		&mode.sg_ospeed,	B0,
	"exta",		&mode.sg_ispeed,	EXTA,
	"exta",		&mode.sg_ospeed,	EXTA,
	"extb",		&mode.sg_ispeed,	EXTB,
	"extb",		&mode.sg_ospeed,	EXTB,
	0
};

struct {
	char *string;
	char *stdname;
} alias[] ={
	"4012",		"tek4012",
	"tek",		"tek4012",
	"tektronix",	"tek4012",
	"cdc",		"cdc713",
	"713",		"cdc713",
	"ti",		"ti700",
	"1620",		"diab1620",
	"diablo",	"diab1620",
	"carousel",	"car300",
	"car",		"car300",
	"1061",		"try1061",
	"teleray",	"try1061",
	"8001",		"int8001",
	"int",		"int8001",
	"intecolor",	"int8001",
	"tty",		"tty43",
	"cops",		"vc404",
	"cops-10",	"vc404",
	"cops10",	"vc404",
	"hazeltine",	"hzl1500",
	"lsi-adm3a",	"adm3a",
	"adm-3a",	"adm3a",
	"vt200",	"vis200",
	"200",		"vis200",
	"1500",		"hzl1500",
	"520",		"ads520",
	"ads520+",	"ads520",
	"520+",		"ads520",
	"LCASE",	"lcase",
	"-LCASE",	"-lcase",
	"UCASE",	"ucase",
	"-UCASE",	"-ucase",
	"DCASE",	"dcase",
	"-DCASE",	"-dcase",
	"interrupt",	"intr",
	"esc",		"escp",
	"escape",	"escp",
	"134.5",	"134",
	0
};

struct {
	char *string;
	short *shptr;
	short shval;
} shval[] ={
	"unknown",	&state.sg_ttype,	UNKNOWN,
	"int8001",	&state.sg_ttype,	INT8001,
	"tek4012",	&state.sg_ttype,	TEK4012,
	"tty43",	&state.sg_ttype,	TTY43,
	"vc404",	&state.sg_ttype,	VC404,
	"try1061",	&state.sg_ttype,	TRY1061,
	"car300",	&state.sg_ttype,	CAR300,
	"ti700",	&state.sg_ttype,	TI700,
	"cdc713",	&state.sg_ttype,	CDC713,
	"diab1620",	&state.sg_ttype,	DIAB1620,
	"la36",		&state.sg_ttype,	LA36,
	"hzl1500",	&state.sg_ttype,	HZL1500,
	"adm3a",	&state.sg_ttype,	ADM3A,
	"ads520",	&state.sg_ttype,	ADS520,
	"vis200",	&state.sg_ttype,	VIS200,
	0
};

struct {
	char *string;
	char *chrp;
} delay[] ={
	"nldel",	&del.sg_nldel,
	"tabdel",	&del.sg_tabdel,
	"crdel",	&del.sg_crdel,
	"ffdel",	&del.sg_ffdel,
	"chdel",	&del.sg_ddel,
	"pgdel",	&pag.sg_pgdel,
	0
};

struct {
	char *string;
	char *chrp;
} numval[] ={
	"width",	&state.sg_width,
	"length",	&state.sg_depth,
	"depth",	&state.sg_depth,
	"pglen",	&pag.sg_pglen,
	0
};

struct {
	char *string;
} reset[] ={
	"reset",
	"tty43",
	"int8001",
	"car300",
	"diab1620",
	"try1061",
	"vc404",
	"ti700",
	"la36",
	"cdc713",
	"tek4012",
	"hzl1500",
	"adm3a",
	"ads520",
	"vis200",
	0
};

struct {
	char *string;
} flush[] ={
	"flush",
	"raw",
	"-raw",
	"cbreak",
	"-cbreak",
	"cooked",
	"9600",
	"4800",
	"2400",
	"1800",
	"1200",
	"600",
	"300",
	"150",
	"110",
	"75",
	"50",
	"0",
	"exta",
	"extb",
	0
};

struct {
	char *string;
	int cmdval;
} xio[] = {
	"hup",		TIOCHPCL,
	"excl",		TIOCEXCL,
	"-excl",	TIOCNXCL,
	"rsto",		TIOCRSTO,
	"zap",		TIOCFLUSH,
	0
};

struct {
	char *string;
	char chval;
} chname[] ={
	"nul",		0,
	"null",		0,
	"soh",		ctl('a'),
	"stx",		ctl('b'),
	"etc",		ctl('c'),
	"eot",		ctl('d'),
	"enq",		ctl('e'),
	"wru",		ctl('e'),
	"ack",		ctl('f'),
	"ru",		ctl('f'),
	"bel",		ctl('g'),
	"bell",		ctl('g'),
	"bs",		ctl('h'),
	"backspace",	ctl('h'),
	"bsp",		ctl('h'),
	"ht",		ctl('i'),
	"tab",		ctl('i'),
	"lf",		ctl('j'),
	"linefeed",	ctl('j'),
	"line-feed",	ctl('j'),
	"newline",	ctl('j'),
	"nl",		ctl('j'),
	"vt",		ctl('k'),
	"ff",		ctl('l'),
	"form",		ctl('l'),
	"formfeed",	ctl('l'),
	"form-feed",	ctl('l'),
	"np",		ctl('l'),
	"cr",		ctl('m'),
	"return",	ctl('m'),
	"so",		ctl('n'),
	"si",		ctl('o'),
	"dle",		ctl('p'),
	"dc1",		ctl('q'),
	"dc2",		ctl('r'),
	"dc3",		ctl('s'),
	"dc4",		ctl('t'),
	"nak",		ctl('u'),
	"syn",		ctl('v'),
	"etb",		ctl('w'),
	"can",		ctl('x'),
	"em",		ctl('y'),
	"sub",		ctl('z'),
	"esc",		ctl('['),
	"escape",	ctl('['),
	"fs",		ctl('\\'),
	"gs",		ctl(']'),
	"rs",		ctl('^'),
	"us",		ctl('_'),
	"space",	' ',
	"sp",		' ',
	"blank",	' ',
	"backslash",	'\\',
	"tilde",	'~',
	"del",		0177,
	"delete",	0177,
	"rubout",	0177,
	"off",		0200,		/* character doesn't work */
	0
};

struct {
	char *string;
	char	prval;
} prname[] ={
	"delete",	0177,
	"backspace",	ctl('h'),
	"tab",		ctl('i'),
	"null",		0,
	"escape",	033,
	"eot",		ctl('d'),
	"space",	' ',
	"OFF",		0200,
	0
};

struct {
	char *on;
	char *off;
} flgname[16] ={
	"tandem",	"-tandem",
	"cbreak",	"-cbreak",
	"lcase",	"-lcase",
	"echo",		"-echo",
	"-nl",		"nl",			/* CRMOD */
	"raw",		"-raw",
	"odd",		"-odd",
	"even",		"-even",
	"ucase",	"-ucase",
	"knl",		"-knl",
	"",		"",			/* RO_TYPE: RO_VDU */
	"",		"",			/* RO_TYPE: RO_HARD */
	"wrap",		"-wrap",
	"page",		"-page",
	"hold",		"-hold",
	"dcase",	"-dcase",
};

struct {
	char *on;
	char *off;
} pgflag[16] ={
	"halt",		"-halt",
	"pause",	"-pause",
	"fixed",	"-fixed",
	"clear",	"-clear",
	"nlgo",		"-nlgo",
};

struct {
	char *string;
	short tyval;
} tynames[] ={
	"Teleray 1061",			TRY1061,
	"Cops-10 (VC404)",		VC404,
	"Intecolor 8001",		INT8001,
	"CDC 713",			CDC713,
	"Telytype 43",			TTY43,
	"Texas Instruments Silent 700",	TI700,
	"Diablo 1620",			DIAB1620,
	"Carousel 300",			CAR300,
	"VT52",				VT52,
	"VT100",			VT100,
	"ADM-3A",			ADM3A,
	"VT05",				VT05,
	"LA36",				LA36,
	"LS120",			LS120,
	"LA180",			LA180,
	"Telytype 37",			TTY37,
	"Tektronix 4010",		TEK4010,
	"Tektronix 4012",		TEK4012,
	"Tektronix 4014",		TEK4014,
	"Tektronix 4015",		TEK4015,
	"Hazeltine 1500",		HZL1500,
	"Visual Technology 200",	VIS200,
	"ADS 520",			ADS520,
	0
};

struct {
	char *string;
	char spd;
} rate[] ={
	"9600",		B9600,
	"4800",		B4800,
	"2400",		B2400,
	"1800",		B1800,
	"1200",		B1200,
	"600",		B600,
	"300",		B300,
	"150",		B150,
	"134.5",	B134,
	"110",		B110,
	"75",		B75,
	"50",		B50,
	"exta",		EXTA,
	"extb",		EXTB,
	"off ?",	B0,
	0
};

struct {
	char *string;
	char dval;
	char *type;
} delname[] ={
	"tty37",	NL_37,		"nldel",
	"tty37",	TAB_37,		"tabdel",
	"delete",	0200,		"chdel",
	"ignore",	0200,		"chdel",
	0
};

int column = 0;		/* current print column */
int vflag  = 0;		/* 'v'(erbose) flag */
int smode = TIOCSETN;	/* ioctl to use ... default is no flush */
int found;		/* != 0 --> have seen a use for this arg */
int vused;		/* != 0 --> have used the value of this arg */
char delim;		/* delimiter for use in printed report */
int nopagemode;		/* != 0 --> tty driver has no page mode installed */

char *arg;		/* the current argument */
char *val;		/* and its value */

#define	NIL	(char *)0

FILE *ofd;		/* output destination (errs always to stderr) */
int tfd;		/* file desc of unit 'stty' is directed at */

char *index();
char *baud();


main(argc, argv)
char **argv;
{
	register i;

	if (!isatty(tfd=1) && !isatty(tfd=0) && !isatty(tfd=2)) {
		fprintf(stderr, "Stty: Can't find a tty to operate on\n");
		exit(2);
	}
	if (tfd == 1)
		ofd = stderr;
	else
		ofd = stdout;

	doioctl(TIOCGETP, (char *)&mode);
	doioctl(TIOCGETC, (char *)&mod);
	doioctl(TIOCGCHR, (char *)&chr);
	doioctl(TIOCGDEL, (char *)&del);
	doioctl(TIOCGETS, (char *)&state);
	doioctl(TIOCGCTL, (char *)&cntl);
	doioctl(TIOCGPAG, (char *)&pag);

	defmode.sg_ispeed = mode.sg_ispeed;
	defmode.sg_ospeed = mode.sg_ospeed;
	defstate.sg_state = state.sg_state;

	if (--argc == 0) {
		prmodes(0);
		exit(0);
	};

	if (argc == 1) {
		if (index(argv[1], '=') == NIL) {
			split(argv[1]);
			if (eq("-v") || eq ("-vv")) {
				prmodes(1);
				exit(0);
			}
			doalias(&arg);
			for (i=0; chars[i].string; i++)
				if (eq(chars[i].string)) {
					prchar(arg, *chars[i].chrp);
					endline();
					exit(0);
				}
			for (i = 0; delay[i].string; i++)
				if (eq(delay[i].string)) {
					prdel(arg, *delay[i].chrp, 1);
					endline();
					exit(0);
				}
		}
	}

	while (--argc >= 0) {
		split(*++argv);
		doalias(&arg);
		if (val == NIL)
			if (eq("-v")) {
				vflag = 1;
				continue;
			} else if (eq("-vv")) {
				vflag = 2;
				continue;
			}
		found = vused = 0;
		if (eq("reset")) {
			mode = defmode;
			mod = defmod;
			chr = defchr;
			del = defdel;
			state = defstate;
			cntl = defcntl;
			pag = defpag;
		}
		for (i = 0; xio[i].string; i++)
			if (eq(xio[i].string))
				doioctl(xio[i].cmdval, (char *)0);
		if (eq("dctl")) {
			if (val == NIL) {
				val = "@ABCDEFHKNOPQRSTUVWXYZ[\\]^_";
				cntl.sg_dctl = 0;
			}
			vused++;
			for (i = 0; val[i]; i++)
				cntl.sg_dctl |= 1 << ctl(val[i]);
		}
		if (eq("-dctl")) {
			if (val == NIL) {
				val = "";
				cntl.sg_dctl = 0;
			}
			vused++;
			for (i = 0; val[i]; i++)
				cntl.sg_dctl &= ~(1 << ctl(val[i]));
		}
		for (i = 0; flush[i].string; i++)
			if (eq(flush[i].string))
				smode = TIOCSETP;
		if (eq("-flush"))
			smode = TIOCSETN;
		for (i = 0; reset[i].string; i++)
			if (eq(reset[i].string)) {
				del = defdel;
				state.sg_width = 0;
				state.sg_depth = 0;
				pag.sg_pglen = 0;
				break;
			}
		for (i = 0; modes[i].string; i++)
			if (eq(modes[i].string)) {
				mode.sg_flags &= ~modes[i].reset;
				mode.sg_flags |=  modes[i].set;
			}
		for (i = 0; pgflags[i].string; i++)
			if (eq(pgflags[i].string)) {
				pag.sg_pgflg &= ~pgflags[i].pgreset;
				pag.sg_pgflg |=  pgflags[i].pgset;
			}
		for (i = 0; chars[i].string; i++)
			if (eq(chars[i].string)) {
				makeval(&argc, &argv);
				*chars[i].chrp = getch();
				break;
			}
		for (i = 0; cval[i].string; i++)
			if (eq(cval[i].string))
				*cval[i].chrp = cval[i].ccval;
		for (i = 0; shval[i].string; i++)
			if (eq(shval[i].string))
				*shval[i].shptr = shval[i].shval;
		for (i = 0; delay[i].string; i++)
			if (eq(delay[i].string)) {
				makeval(&argc, &argv);
				*delay[i].chrp = getdel();
			}
		for (i = 0; numval[i].string; i++)
			if (eq(numval[i].string)) {
				makeval(&argc, &argv);
				*numval[i].chrp = getnum();
			}
		if (!found) {
			fprintf(stderr, "Stty: Unknown mode %s", arg);
			if (val != NIL)
				fprintf(stderr, "=%s", val);
			putc('\n', stderr);
			exit(1);
		}
		if (val != NIL && !vused)
			fprintf(stderr, "Stty: Value %s for %s ignored\n",
					val, arg);
	}
	doioctl(smode, (char *)&mode);
	doioctl(TIOCSETC, (char *)&mod);
	doioctl(TIOCSCHR, (char *)&chr);
	doioctl(TIOCSDEL, (char *)&del);
	doioctl(TIOCSETS, (char *)&state);
	doioctl(TIOCSCTL, (char *)&cntl);
	if (!nopagemode)
		doioctl(TIOCSPAG, (char *)&pag);

	if (vflag != 0)
		prmodes(vflag-1);
	exit(0);
}

eq(s)
char *s;
{
	if (strcmp(arg, s) == 0) {
		found++;
		return(1);
	}
	return(0);
}

split(s)
char *s;
{
	register char *p;

	arg = s;
	if ((p = index(s, '=')) != NIL) {
		*p = '\0';
		val = p+1;
	} else
		val = NIL;
}

makeval(argc, argv)
int *argc;
char ***argv;
{
	vused++;
	if (val != NIL)
		return;
	if (--*argc >= 0)
		val = *++*argv;
	else {
		fprintf(stderr, "No value specified for %s\n", arg);
		exit(1);
	}
}

getdel()
{
	register i;
	register neg = 0;

	if (val[0] == '-')
		neg++, val++;
	if (isdigit(val[0])) {
		i = ms(getnum());
		if (i > 0177) {
			i = 0177;
			fprintf(stderr, "Stty: Enormous value for %s shortened\n",
					arg);
		}
		if (neg)
			i |= 0200;
		return(i);
	} else {
		if (neg)
			val--;
		doalias(&val);
		for (i = 0; delname[i].string; i++)
			if (equ(delname[i].string, val) && equ(delname[i].type, arg))
				return(delname[i].dval);
		fprintf(stderr, "Stty: Don't know %s for a %s\n", arg, val);
		exit(1);

		/*NOTREACHED*/
	}
}

getch()
{
	register char c, *p;
	register i;

	c = val[0];
	p = &val[1];
	if (c == '^' && *p != '\0' && p[1] == '\0')
		return(ctl(*p));
	if (c == '\\' && val[2] == '\0') switch(*p) {
		case 'n': return('\n');
		case 'f': return('\f');
		case 'b': return('\b');
		case 't': return('\t');
		case 'r': return('\r');
	}
	if (isdigit(c)) {
		if (c != '0')
			return(getnum());
		if (*p == 'x' || *p == 'X')
			return(scan(val+2, "%x"));
		return(scan(val, "%o"));
	}
	for (i = 0; chname[i].string; i++)
		if (equ(chname[i].string, val))
			return(chname[i].chval);
	doalias(&val);
	for (i = 0; chars[i].string; i++)
		if (equ(chars[i].string, val))
			return(*chars[i].chrp);
	if (val[1] == '\0')
		return(c);
	fprintf(stderr, "Stty: Invalid value (%s) for %s\n", val, arg);
	exit(1);
	/*NOTREACHED*/
}

getnum()
{
	return(scan(val, "%d"));
}

scan(str, fmt)
char *str;
char *fmt;
{
	char buf[20];
	int i;
	char c;

	strcat(strcpy(buf, fmt), "%c");
	if (sscanf(str, buf, &i, &c) != 1) {
		fprintf(stderr, "Stty: Bad numeric val (%s) for %s\n", val, arg);
		exit(1);
	}
	return(i);
}

doioctl(cmd, arg)
char *arg;
{
	if (ioctl(tfd, cmd, arg) == -1) {
		if (cmd == TIOCGPAG)
			nopagemode++;
		else {
			perror("Stty");
			exit(1);
		}
	}
}

prchar(s, c)
register c;
char *s;
{
	char *namptr = "", ctlbuf[16], valbuf[16];
	register cch;
	register i, sep = 1;

	c &= 0377;
	ctlbuf[0] = valbuf[0] = '\0';
	if (c >= ' ' && c <= '~')
		sprintf(valbuf, CHFMT, c);
	else if (c <= ctl('z')) {
		sprintf(valbuf, CTLFMT, pchar(c));
		cch = 0;
		switch ( c ) {
			case '\n': cch = 'n'; break;
			case '\f': cch = 'f'; break;
			case '\b': cch = 'b'; break;
			case '\t': cch = 't'; break;
			case '\r': cch = 'r'; break;
		}
		if ( cch ) {
			ctlbuf[1] = '\\';
			ctlbuf[2] = cch;
			ctlbuf[0] = ctlbuf[3] = '\'';
			ctlbuf[4] = 0;
		}
	} else {
		if (c < ' ') {
			sep++;
			sprintf(ctlbuf, CTLFMT, pchar(c));
		}
		sprintf(valbuf, OCTFMT, c);
	}

	for (i = 0; prname[i].string; i++)
		if ((prname[i].prval&0377) == c) {
			namptr = prname[i].string;
			sep++;
			break;
		}

	linefit(strlen(s)+strlen(namptr)+strlen(ctlbuf)+strlen(valbuf)+sep);
	fprintf(ofd, "%s ", s);
	if (*namptr)
		fprintf(ofd, "%s ", namptr);
	if (ctlbuf[0])
		fprintf(ofd, "%s ", ctlbuf);
	fprintf(ofd, "%s", valbuf);
}

linefit(n)
{
	if (column > 0) {
		if (column + n > WIDTH-2)
			endline();
		else {
			if (delim != '\0') {
				putc(delim, ofd);
				column++;
			}
			putc(' ', ofd);
			column++;
		}
	}
	column += n;
}

endline()
{
	if (column > 0)
		putc('\n', ofd);
	column = 0;
}

	/* VARARGS */
xprintf(fmt, a1, a2, a3, a4)
char *fmt;
{
	char buf[80];

	if (fmt == NIL || *fmt == '\0')
		return;

	sprintf(buf, fmt, a1, a2, a3, a4);
	linefit(strlen(buf));
	fputs(buf, ofd);
}

prmodes(v)
{
	register i, j;

	if ((i = state.sg_ttype) || v) {
		for(j = 0; tynames[j].string; j++)
			if (tynames[j].tyval == i)
				break;
		if (tynames[j].string)
			xprintf(tynames[j].string);
		else
			xprintf("Unknown terminal");
	}
	delim = ';';
	if (v) {
		if (state.sg_width != 0 && state.sg_depth != 0)
			xprintf("Page %d x %d", state.sg_width, state.sg_depth);
		else if (state.sg_width != 0)
			xprintf("Width %d", state.sg_width);
		else if (state.sg_depth != 0)
			xprintf("Depth %d", state.sg_depth);
		if (pag.sg_pglen && pag.sg_pglen != state.sg_depth)
			xprintf("Pglen %d", pag.sg_pglen);
	}
	if (mode.sg_ispeed == mode.sg_ospeed)
		xprintf("%s baud", baud(mode.sg_ispeed));
	else
		xprintf("ispeed %s, ospeed %s", baud(mode.sg_ispeed),
						baud(mode.sg_ospeed));
	endline();

	for (i = 1; chars[i].string; i++)
		if (v || *chars[i].chrp && *chars[i].chrp != chars[i].defprt)
			prchar(chars[i].string, *chars[i].chrp);
	endline();
	delim = '\0';

	for (i = 0, j = 1; i < 16; i++, j <<= 1) {
		if (mode.sg_flags & j)
			xprintf(flgname[i].on);
		else if (v)
			xprintf(flgname[i].off);
	}
	if ((del.sg_tabdel&0377) == XTABS)
		xprintf("-tabs");
	else if (v && del.sg_tabdel == 0)
		xprintf("tabs");
	if ((del.sg_ffdel&0377) == XFORM)
		xprintf("-form");
	else if (v && del.sg_ffdel == 0)
		xprintf("form");
	switch (mode.sg_flags & RO_TYPE) {
		case RO_VDU:
			xprintf("vdu");
			break;
		case RO_HARD:
			xprintf("ero");
			break;
		default:
			if (v)
				xprintf("-vdu -ero");
			break;
	}
	if (!(state.sg_state & HUPCLS))
		xprintf("-hup");
	else if (v)
		xprintf("hup");
	if (!nopagemode)
		for (i = 0, j = 1; i < 16; i++, j <<= 1) {
			if (pag.sg_pgflg & j)
				xprintf(pgflag[i].on);
			else if (v)
				xprintf(pgflag[i].off);
		}
	if (v)
		prdctl();
	endline();
	delim = ';';

	prdel("nldel", del.sg_nldel, v);
	prdel("crdel", del.sg_crdel, v);
	prdel("ffdel", del.sg_ffdel, v);
	prdel("tabdel", del.sg_tabdel, v);
	prdel("pgdel", pag.sg_pgdel, v);
	if ((i = del.sg_ddel) != 0) {
		if (i & 0200) {
			prchar("dchar (deleted)", del.sg_dchar);
			i &= 0177;
		} else if (v || del.sg_dchar)
			prchar("dchar", del.sg_dchar);
		prdel("special delay", i, v);
	}
	endline();
}

prdctl()
{
	char buf[40];		/* more than 32 */
	register i, j;
	register char *p;

	for (p = buf, i = 0, j = 1; i < 32; i++, j <<= 1) 
		if (cntl.sg_dctl & j)
			*p++ = pchar(i);
	if (p == buf) {
		xprintf("-dctl");
		return;
	}
	*p = 0;
	xprintf("dctl='%s'", buf);
}

prdel(s, del, v)
register del;
char *s;
{
	register i;

	if (del & 0200) {
		del &= 0377;
		for (i = 0; delname[i].string; i++)
			if ((delname[i].dval&0377) == del
				&& equ(delname[i].type, s) ) {
					xprintf("%s=%s", s, delname[i].string);
					return;
			}
		if (del != 0200)
			xprintf("%s strange (%.1o)", s, del);
		return;
	}
	if (v || del)
		xprintf("%s = %d ms", s, (del*1000)/HZ);
}

char *
baud(spd)
{
	register i;

	for (i = 0; rate[i].string; i++)
		if (spd == rate[i].spd)
			return(rate[i].string);
	return("??");
}

doalias(str)
register char **str;
{
	register i;

	for (i = 0; alias[i].string; i++)
		if (equ(alias[i].string, *str)) {
			*str = alias[i].stdname;
			break;
		}
}
