/*	errlog.c	M01	Jun 1 80 */

#include "../h/opts/ERR_LOG.h"
#ifdef	ERR_LOG
/*
 *	Error logging
 */

#include	"../h/param.h"
#include	"../h/proc.h"
#include	"../h/buf.h"
#include	"../h/dir.h"
#include	"../h/user.h"
#include	"../h/errlog.h"

/*
 *	Each device (etc) that wishes to log errors must provide
 *	one or more errec structures in which to record device
 *	dependant error info. The first element of the structure
 *	should be an errhead structure (see ../h/errlog.h)
 *
 *	The address of such a structure, together with its size
 *	(which is variable) are passed to logerr, which records
 *	this info until read by some user process. Each read by
 *	such a process will return one error record, or an EOF
 *	indication to indicate a particularly high error rate
 *	causing errors to be lost. If there is no reader, errors
 *	are not recorded.
 *
 *	The device must not re-use the error structure till the
 *	EH_BUSY bit in the header flags is reset
 */

struct	errrec	{	/* a prototype */
	struct	errhead	er_h;
	char	er_d[1];
};

/*
 * info on pending errors is retained here
 */

#define	ERRFAST	28	/* when # pend errs > this, reader niced very -'ve */
#define	ERRHIGH	20	/* when # pend errs > this, reader niced -'ve */
#define	ERRLOW	6	/* when # pend errs < this, reader niced 0 */

#define	FASTNICE	(NZERO-18)
#define	HIGHNICE	(NZERO-8)
#define	LOWNICE		NZERO

#define	ERRPRI		(PZERO+1)	/* sleep pri, should be > PZERO
					   so can be broken by signals	*/

struct	errrec	*errhead = NULL;	/* first error on queue */
struct	errrec	*errtail = NULL;	/* last error on queue */

int	errcnt;				/* # errors pending */

struct	proc	*errproc;		/* u.u_procp of error logger */

int	errflags;
#define	ERROPEN		01		/* there is an errlogger */
#define	ERRLOST		02		/* there have been lost errors */
#define	ERRSLEEP	04		/* err logger is waiting for errs */


/*
 *	This routine is called by device routines to log errors
 */
logerr(ep)
register struct	errrec	*ep;
{
	if ( (errflags & ERROPEN) == 0 ) {
		errflags |= ERRLOST;
		return(0);
	}

	errcnt++;
	ep->er_flags |= EH_BUSY;
	ep->er_nxt = NULL;
	if (errtail != NULL)
		errtail->er_nxt = ep;
	errtail = ep;
	if (errhead == NULL)
		errhead = ep;

	if ( errcnt > ERRHIGH ) {
		if ( errcnt > ERRFAST )
			errproc->p_nice = FASTNICE;
		else
			errproc->p_nice = HIGHNICE;
		setpri(errproc);
	}

	if (errflags & ERRSLEEP) {
		errflags &= ~ERRSLEEP;
		wakeup((caddr_t)&errhead);
	}

	return(1);
}


/*
 *	This lot implements the errlog device
 */

errlopen(dev, flag)
dev_t	dev;
{
	if (errflags & ERROPEN) {
		u.u_error = EBUSY;
		return;
	}

	if (flag & 02) {
		u.u_error = ENXIO;
		return;
	}

	(errproc = u.u_procp) -> p_nice = LOWNICE;
		/* (we can afford to wait for a natural setpri here) */

	errcnt = 0;
	errflags = ERROPEN | (errflags & ERRLOST);
	errhead = NULL;
	errtail = NULL;
}

errlclose(dev, flag)
dev_t	dev;
{
	register struct errrec *ep;

	if (errcnt != 0)
		errflags |= ERRLOST;
	errflags &= ~(ERROPEN | ERRSLEEP);
	for (ep = errhead; ep; ep = ep->er_nxt)
		ep->er_flags &= ~EH_BUSY;
	errhead = NULL;
	errtail = NULL;
}

errread(dev)
dev_t	dev;
{
	register struct errrec *ep;

	if (errflags & ERRLOST) {
		errflags &= ~ERRLOST;
		return;			/* will give EOF status, 0 len rd */
	}

	while (errcnt == 0) {
		errflags |= ERRSLEEP;
		sleep((caddr_t)&errhead, ERRPRI);
	}

	ep = errhead;

	iomove((caddr_t)ep->er_d, min(u.u_count, ep->er_size), B_READ);

	VOID spl6();
	errhead = ep->er_nxt;
	if (errhead == NULL)
		errtail = NULL;
	VOID spl0();
	ep->er_flags &= ~EH_BUSY;

	if (--errcnt < ERRLOW) {
		errproc->p_nice = LOWNICE;
		setpri(errproc);
	}
}
#endif
