/* netstat.c: show network status for internet networks */
#include <stdio.h>
#include <nlist.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/inet/in.h>
#include <sys/inio.h>
#include <sys/inet/ip_var.h>
#include <sys/inet/udp_user.h>
#include <sys/inet/tcp_user.h>
#include <sys/inet/tcp.h>

struct	nlist	names[] = {
#define	SIPIF	0
	{ "_ipif" },
#define	SIPDEV	1
	{ "_ipdev" },
#define	SUDPDEV 2
	{ "_udpdev" },
#define STCPDEV 3
        { "_tcpdev" },
#define	SROUTE	4
	{ "_rte_tab" },
#define	SARP	5
	{ "_arptab" },
#define	STCB	6
	{ "_tcb" },
	{ NULL }
};

int	arpflag;	/* print known mapping between internet and ether */
int	conflag;	/* all TCP and UDP connections */
int	tcpflag;	/* detailed state of active TCP connections */
int	ipflag;		/* active IP interfaces */
int	nflag;		/* display numeric internet addresses */
int	statflag;	/* protocol statistics */
int	routeflag;	/* routing tables */
int	allrflag;	/* routeflag including deleted entries */

char	*core	= "/dev/kmem";
char	*system	= "/unix";
int	corfd;

char	*progname;
char 	*hname();
char	*pr_udp_state();

main(argc, argv)
	int argc;
	char *argv[];
{
	register char *cp;

	progname = argv[0];
	argc--;
	argv++;
	if (argc == 0)
		conflag++;
	else if (**argv == '-') {
		for (cp = &argv[0][1]; *cp; cp++)
			switch (*cp) {
			case 'a':	arpflag++;	break;
			case 'c':	conflag++;	break;
			case 'C':	tcpflag++;	break;
			case 'i':	ipflag++;	break;
			case 'n':	nflag++;	break;
			case 's':	statflag++;	break;
			case 'r':	routeflag++;	break;
			case 'R':	allrflag++;	break;
			case 't':	;		break;
			default:	usage();	break;
			}
		argv++;
		argc--;
	}
	if (argc > 0) {
		system = *argv;
		argv++;
		argc--;
	}
	if (argc > 0) {
		core = *argv;
		argv++;
		argc--;
	}
	nlist(system, names);
	if (names[SIPIF].n_type == 0)
		error("%s has no namelist", system);
	corfd = eopen(core, 0);
	if (arpflag)
		doarp();
	if (ipflag)
		doip();
	if (conflag)
		docon();
	if (routeflag)
		dorte();
	return 0;
}


usage()	/* print usage and die */
{
	fprintf(stderr, "usage: netstat [-acCirRst][system][core]\n");
	exit(1);
}
eopen(name, how)
	char *name;
{
	register fd;

	if ((fd = open(name, how)) == -1)
		error("open failed: %s", name);
	return fd;
}

eioctl(fd, cmd, arg)	/* error version of ioctl */
	int fd;
	int cmd;
	char *arg;
{
	if (ioctl(fd, cmd, arg) == -1)
		error("ioctl 0x%x failed", (char *)cmd);
}

error(s1, s2)
	register char *s1, *s2;
{
	extern int errno, sys_nerr;
	extern char *sys_errlist[], *progname;

	if (progname)
		fprintf(stderr, "%s: ", progname);
	fprintf(stderr, s1, s2);
	if (errno > 0 && errno < sys_nerr)
		fprintf(stderr, " (%s)", sys_errlist[errno]);
	fprintf(stderr, "\n");
	exit(1);
}

doarp()	/* print arp table from kernel */
{
	struct arp arps[NARP];
	register struct arp *ap;
	register loc, count = 0;

	loc = names[SARP].n_value;
	kseek(corfd, loc, 0);
	read(corfd, (char *)arps, sizeof arps);
	printf ("Arp table\n");
	printf ("%-8s %-18s %s\n", "LOC", "ETHERNET", "INTERNET ADDRESS");
	for (ap = arps; ap < &arps[NARP]; ap++) {
		if (ap->inaddr) {
			count++;
			printf("%08x ", loc);
			prenet(ap->enaddr);
			putchar(' ');
			printf("%s\n", hname(ap->inaddr));
		}
		loc += sizeof (struct arp);
	}
	printf ("%d/%d used\n\n", count, NARP);
}

prenet(enet)
	u_char enet[6];
{
	register n = 6;
	register u_char *p = enet;

	while (n--)
		printf("%02x%c", *p++, n > 0 ? ':' : ' ');
}

kseek(fd, offset, how)
{
	return lseek(fd, offset & ~0xc0000000, how);
}


doip()	/* print active IP interfaces */
{
	struct ipif ipif[NIPLD];
	register struct ipif *p;
	register loc, count = 0;
	char buf[20];
	
	loc = names[SIPIF].n_value;
	kseek(corfd, loc, 0);
	read(corfd, (char *)ipif, sizeof ipif);
	printf("%-8s %-15s %-15s %-15s\n", "LOC", "QUEUE", "FLAGS", "MTU");
	printf("\t %-15s %-15s %-15s %-15s\n",
			"THIS HOST", "THAT", "MASK", "BROADCAST");
	printf("\t %-15s %-15s %-15s %-15s\n", 
		"IPACKETS", "IERRORS", "OPACKETS", "OERRORS");
	printf("\t %-15s %-15s\n\n", "ARP", "DEVICE");
	for (p = ipif; p < &ipif[NIPLD]; p++, loc += sizeof (struct ipif)) {
		if (p->queue == NULL)
			continue;
		count++;
		printf("%08x ", loc);
		printf("%08x%7s ", p->queue, "");
		printf("%04o%11s ", p->flags, "");
		printf("%4d%\n", p->mtu);
		printf("\t ");
		printf("%-15.15s ", hname(p->thishost));
		printf("%-15.15s ", hname(p->that));
		printf("%-15.15s ", in_ntoa(p->mask));
		printf("%-15.15s \n", in_ntoa(p->broadcast));
		printf("\t ");
		printf("%-15d %-15d ", p->ipackets, p->ierrors);
		printf("%-15d %-15d\n", p->opackets, p->oerrors);
		printf("\t ");
		printf("%3s%12s ", p->arp ? "arp" : "" , "");
		sprintf(buf, "(%d,%d)", major(p->dev), minor(p->dev));
		printf("%-15s\n", buf);
	}
	printf("%d/%d used\n\n", count, NIPLD);
}

char *
hname(addr)	/* convert to string based on nflag */
	in_addr addr;
{
	return nflag ? in_ntoa(addr) : in_host(addr);
}

docon()		/* display connection assignments for TCP and UDP */
{
	printf("DEV    %-25s%-25sSTATE\n", "LOCAL", "FOREIGN");
	tcpcon();
	udpcon();
}

tcpcon()
{
	struct tcb tcb[NTCB];
	register struct tcb *tp;
	register loc, count = 0;
	char buf[80];

	loc = names[STCB].n_value;
	kseek(corfd, loc, 0);
	read(corfd, (char *)tcb, sizeof tcb);
	for (tp = tcb; tp < &tcb[NTCB]; tp++) {
		if (tp->state == CLOSED)
			continue;
		printf("tcp%02d  ", tp - &tcb[0]);
		buf[0] = '\0';
		pr_addr(buf, tp->laddr);
		pr_port(buf, tp->lport, "tcp");
		printf("%-25s", buf);
		buf[0] = '\0';
		pr_addr(buf, tp->faddr);
		pr_port(buf, tp->fport, "tcp");
		printf("%-25s", buf);
		pr_tcb_state(tp->state);
		printf("\n");
	}
}

udpcon()	/* display the udp connections */
{
	struct udpdev udpdev[NUDPDEV];
	register struct udpdev *p;
	register loc, count = 0;
	char buf[30];

	loc = names[SUDPDEV].n_value;
	kseek(corfd, loc, 0);
	read(corfd, (char *)udpdev, sizeof udpdev);
	for (p = udpdev; p < &udpdev[NUDPDEV]; p++, loc += sizeof(struct udpdev)) {
		if (p->state == 0)
			continue;
		count++;
		printf("upd%02d  ", p - udpdev);
		buf[0] = '\0';
		pr_addr(buf, p->laddr);
		pr_port(buf, p->lport, "udp");
		printf("%-25s", buf);
		buf[0] = '\0';
		pr_addr(buf, p->faddr);
		pr_port(buf, p->fport, "udp");
		printf("%-25s", buf);
		printf("%-9s ", pr_udp_state(p->state));
		printf("\n");
	}
}

char *
pr_udp_state(state)	/* return pointer to ascii state */
{
	static char buf[40];

	switch (state) {
	case 0:	return "closed";
	case 1: return "unbound";
	case 2: return "datagram";
	case 3: return "connected";
	case 4: return "listening";
	default:
		sprintf(buf, "%d", state);
		return buf;
	}
}

dorte()	/* print route tables */
{
	struct route r[NROUTE];
	register struct route *p;
	int count = 0;

	kseek(corfd, names[SROUTE].n_value, 0);
	read(corfd, (char *)r, sizeof r);
	printf("ROUTE\n");
	printf("%-20s%-20s\n", "DEST", "GATE");
	for (p = r; p < &r[NROUTE]; p++)
		if (p->dst) {
			printf("%-20s", hname(p->dst));
			printf("%-20s\n", hname(p->gate));
			count++;
		}
	printf("%d/%d used\n", count, NROUTE);
}
pr_tcb_state(state)
{
	static char *sn[] = {
		"closed",
		"listen", "syn_sent", "syn_received", "established", "fin_wait_1",
		"fin_wait_2", "close_wait", "closing", "last_ack", "time_wait",
	};
	if (state < 0 || state > TIME_WAIT)
		printf("UNKNOWN");
	else
		printf("%s ", sn[state]);
}

pr_addr(b, a)		/* catenate ascii for address */
	char *b;
	in_addr a;
{
	while (*b)
		b++;
	if (nflag) {
		sprintf(b, "%s:", in_ntoa(a));
		return;
	}
	sprintf(b, "%s:", in_host(a));
}

pr_port(b, p, proto)		/* catenate ascii for port */
	char *b, *proto;
	unsigned long p;
{
	register struct in_service *ip;

	while (*b)
		b++;
	if ((ip = in_service((char *)0, proto, p)) == NULL || nflag)
		sprintf(b, "%d", p);
	else
		sprintf(b, "%s", ip->name);
}
