/* scsi.c:  Driver for wd3393 scsi controller chip */
/*	Scsi() will queue one request if busy and issue it after
 *	the interrupt handler for the current request is called
 *	and hasn't made another request.  This means that chains
 *	of transactions don't get broken.
 */


#include "param.h"
#include "types.h"
#include "buf.h"
#include "reboot.h"

#define	SCSIADDR	((u_char *) 0xfffe4000)
#define	SCSIDATA	((u_char *) 0xfffe4001)
#define	SBT		0x80

#define	get(a)		(*SCSIADDR = (a), *SCSIDATA)
#define	set(a, d)	(*SCSIADDR = (a), *SCSIDATA = d)

static
struct	wd3393	{
	u_char	control;		/* enable/disable certain functions */
	u_char	timeout;		/* = (Tper * Ficlk) / 80 */
	u_char	cdb[12];		/* the command to send */
	u_char	target_lun;		/* set during reconnect */
	u_char	cmd_phase;		/* where the chip is in things */
	long	xfer_cnt;		/* bytes to move */
	u_char	dest_id;		/* scsi destination register */
	u_char	src_id;			/* who am i */
	u_char	status;			/* chip status */
	u_char	command;		/* chip command */
} d;

static	busy;
static	next_waiting;
static	int next_id, next_flags, next_count;
static	char next_cdb[12];
static	caddr_t	next_addr;
static	int	(*next_intr)();
static	int	scsi_inited;

static	int	(*inthandler)();	/* upper level int handler */


scsi(id, flags, cdb, addr, count, intfunc)		/* do scsi cmd */
	char *cdb;
	caddr_t addr;
	int (*intfunc)();
{
	register u_char *fp;
	register s, i;

	s = splbio();
	if (busy) {
		next_id = id;
		next_flags = flags;
		bcopy(cdb, next_cdb, 12);
		next_addr = addr;
		next_count = count;
		next_intr = intfunc;
		next_waiting = 1;
		return;
	} else
		busy = 1;
	splx(s);
	if (! scsi_inited) {		/* this never happens as on interrupt */
		printf ("initing scsi chip\n");
		resetscsi(1);		/* assert reset line */
		delay(100);
		resetscsi(0);		/* de-assert scsi reset line */
		delay(2000);
		set(0, 7);		/* scsi id 7, no par, 10MHz */
		delay(2);
		set(0x18, 0);		/* reset the chip */
		sleep(&inthandler, PRIBIO);		/* okay to sleep */
		scsi_inited = 1;
	}
	inthandler = intfunc;
	if (count) 
		setdma(addr, count, (flags & B_READ));
	d.control = 0x84;
	d.timeout = 62;		/* 500 ms */
	bcopy(cdb, d.cdb, sizeof d.cdb);
	if (id == 4)
	d.target_lun = 0;
	d.cmd_phase = 0;
	d.xfer_cnt = count & 0xffffff;
	d.dest_id = id;
	d.status = 0;
	d.command = 9;		/* select and transfer w/o atn */
	delay(2);
	*SCSIADDR = 1;		/* start with the control register */
	fp = (u_char *)&d;
	for (i = 0; i < ((char *)&d.command - (char *)&d)+1; i++)
		*SCSIDATA = *fp++;
}

pccscsiportintr()		/* catch the chip's interrupt */
{
	register status;
	register (*handler)();

	status = get(0x17);
	switch (status) {
	case 0:				/* reset */
	case 1:
		wakeup(&inthandler);
		break;
#ifdef notdef
	case 0x4B:		/* unexpected phase -- status */
		delay(2);
		printf ("pccscsiportint: 0x%x\n", status);
		set(0x18, 0x20 | SBT);	/* transfer it in */
		break;
#endif
	case 0x85:		/* target disconnect -- all done */
		busy = 0;
		handler = inthandler;
		if (next_waiting) {
			next_waiting = 0;
			scsi(next_id, next_flags, next_cdb, next_addr, 
				next_count, next_intr);
		}
		if (handler != NULL)
			(*handler)(0);		/* call user int handler */
		else
			printf ("pccscsiintr: inthandler NULL!\n");
		break;
	case 0x16:		/* sel&xfer complete */
/*		delay(2000);		/* does this help? */
		delay(45);
		break;
	default:
		printf("pccscsiintr: status 0x%x\n", status);
		resetscsi(1);		/* assert reset line */
		delay(100);
		resetscsi(0);		/* de-assert scsi reset line */
		pcc_reboot(RB_NOSYNC);
		clrdma();
		scsi_inited = 0;
		busy = 0;
		handler = inthandler;
		if (next_waiting) {
			next_waiting = 0;
			scsi(next_id, next_flags, next_cdb, next_addr, 
				next_count, next_intr);
		}
		if (handler != NULL)
			(*handler)(1);
	}
}
