/* zi.c: Driver for zilog 8530 chip. */

#include "param.h"
#include "tty.h"
#include "dir.h"
#include "signal.h"
#include "seg.h"
#include "ipm.h"
#include "sid.h"
#include "user.h"
#include "errno.h"
#include "buf.h"
#include "systm.h"


#define	SCC	((struct scc *) 0xfffe3000)
#define	set(r, x)	(SCC->a_wr0 = r, SCC->a_wr0 = x)
#define	get(r)		(SCC->a_wr0 = (r), SCC->a_wr0)
#define	bget(r)		(SCC->b_wr0 = (r), SCC->b_wr0)
#define	bset(r, x)	(SCC->b_wr0 = (r), SCC->b_wr0 = (x))
#define	VECTOR	0x80 
#define	B9600	13

struct	scc	{
	char	b_wr0;
	char	b_tdr;		/* xmit and rcv register */
	char	a_wr0;		/* control/status byte */
	char	a_tdr;		/* xmit and rcv register */
};
#define	b_rdr	b_tdr
#define	b_rr0	b_wr0
#define a_rdr	a_tdr
#define	a_rr0	a_wr0

static	int	reg5;			/* fake register that can be read */
static	int	reg1;			/*  ditto */

static	int speeds[] = {
	-1,	/* hang up phone */
	-1,	/* 50 baud */
	-1,	/* 75 baud */
	-1,	/* 110 baud */
	-1,	/* 134.5 baud */
	-1,	/* 150 baud */
	-1,	/* 200 baud */
	518,	/* 300 baud */
	258,	/* 600 baud */
	128,	/* 1200 baud */
	-1,	/* 1800 baud */
	63,	/* 2400 baud */
	31,	/* 4800 baud */
	14,	/* 9600 baud */
	6,	/* Exta (19.2) baud */
	2,	/* Extb (38.4k) */
};

/*
 *	The order of initialization is very critical to this dippy chip.
 *	This order was specified in the Advanced Micro Devices Technical
 *	manual ``Serial Communications Controller AmZ8030/8530.''
 *	Order #07513a.
 *	page 8-3.
 */

struct	tty	zitty[4];
static	int	zistart();
int	ttrstrt();

ziopen(dev, flag)		/* open the b port as a tty */
	dev_t dev;
{
	register r, unit;
	register struct tty *tp;

	if ((unit = minor(dev)) > 0) {
		u.u_error = ENXIO;
		return;
	}
	tp = &zitty[unit];
	if ((tp->t_state & (ISOPEN|WOPEN)) == 0) {
		tp->t_oproc = zistart;
		tp->t_flags = ANYP | ECHO | CRMOD;
		tp->t_ispeed = tp->t_ospeed = B9600;
		SCC->a_wr0 = 0x30;	/* clear receiver error status */
		SCC->a_wr0 = 0x10;	/* clear ext status interrupts */
		set(9, 0xC0);		/* hard reset */
		set(4, 0x4c);		/* /16, nopar, 1 stop */
		set(2, VECTOR);		/* interrupt vector */
		set(3, 0xC0);		/* 8bits, rcv enabled */
		reg5 = 0x62;
		set(5, reg5);		/* Xmit: 8bits, xmit enab, RTS  */
		set(9, 0x01);		/* vis on */
		set(11, 0x56);		/* use BRG */
		set(12, 14);		/* lower byte of time constant */
		set(13, 0);		/* high byte of time constant */
		set(14, 0x02);		/* BRG gen. clk src = RTXC pin */
		set(14, 0x82);		/* clk = br gen */
		set(14, 0x03);		/* br generator enable */
		set(3, 0xC1);		/* rx enable */
		reg5 = 0x6a;
		set(5, reg5);		/* tx enable */
		set(15, 0);		/* nothing special yet */
		SCC->a_wr0 = 0x10;	/* Reset Extern Status */
		SCC->a_wr0 = 0x10;	/* twice */
		spltty();
		set(1, reg1 = 0x10);	/* int on rx */
		spl0();
		set(9, 0x09);		/* master interrupt enable */
		if (SCC->a_rr0 & 0x08)
			tp->t_state = CARR_ON;
		pcc_tty_intr_on();
		ttychars(tp);
	}
	zimodem(dev, 1);
	tp->t_state |= CARR_ON;
	ttyopen(dev, tp);
}

/*
 *  Let the dtr stay down for two seconds
 * when closing the dev 1.  This will give the modem time
 * to reset DCD.
 */
ziclose(dev, flag)
	dev_t dev;
{
	register struct tty *tp;
	register int r;
	int nap;		/* to sleep for two seconds */

	tp = &zitty[minor(dev)];
	ttyclose(tp);
	set(5, 0);		/* reset DTR and others */
	tp->t_flags = 0;
	tp->t_state = 0;
}

ziread(dev)
	dev_t dev;
{
	ttread(&zitty[minor(dev)]);
}

ziwrite(dev)
	dev_t dev;
{
	ttwrite(&zitty[minor(dev)]);
}

zixintr(dev)
	dev_t dev;
{
	set(1, reg1 = 0x10);		/* disable transmitter */
	SCC->a_wr0 = 0x38;	/* reset highest ius */
	ttstart(&zitty[minor(0)]);
}

zisrintr()		/* special receive interrupt */
{
	register x;

	x = SCC->a_rdr;
	SCC->a_wr0 = 0x30;		/* error reset */
	SCC->a_wr0 = 0x38;		/* reset highest ius */
}

zieintr(dev)
	dev_t dev;
{
	register struct tty *tp;
	int x;

	tp = &zitty[minor(0)];
	SCC->a_wr0 = 0x10;		/* reset external status */
	SCC->a_wr0 = 0x38;		/* reset highest ius */
	x = SCC->a_rdr;			/* read null character */
}

zirintr(dev)
	dev_t dev;
{
	register char c;
	register struct tty *tp;

	tp = &zitty[minor(0)];
	while (SCC->a_rr0 & 01) {
		c = SCC->a_rdr;
		ttyinput(c, tp);
	}
	SCC->a_wr0 = 0x38;		/* reset highest IUS */
}

ziioctl(dev, cmd, addr, flag)
	caddr_t addr;
	dev_t dev;
{
	int r;
	register struct tty *tp;

	tp = &zitty[minor(dev)];
	if (ttioccomm(cmd, tp, addr, dev) == 0) {
		u.u_error = ENOTTY;
		return;
	}
	if (cmd == TIOCSETP) {
		r = speeds[tp->t_ispeed];
		if (r != -1)
			set(12,r);
	}
}

static
zistart(tp)
	register struct tty *tp;
{
	register int c;
	int s;

	s = spltty();
	if ((SCC->a_rr0 & 04) == 0)
		return;
	if ((c = getc(&tp->t_outq)) >= 0) {
		set(1, reg1 = 0x12);
		if (tp->t_flags & RAW) {
			SCC->a_tdr = c;
		} else if (c <= 0177)
			SCC->a_tdr = c;
		else {
			timeout(ttrstrt, (caddr_t)tp, (c & 0177) + 6);
			tp->t_state |= TIMEOUT;
		}
		if (tp->t_outq.c_cc <= TTLOWAT && tp->t_state & ASLEEP) {
			tp->t_state &= ~ASLEEP;
			wakeup((caddr_t) & tp->t_outq);
		}
	}
	splx(s);
}

static
zimodem(dev, flag)		/* modem control */
	dev_t	dev;
{
	if (flag)
		set(5, reg5 |= 0x80);	/* turn on dtr */
	else
		set(5, reg5 &= ~0x80);	/* turn off dtr */
}

char *msgbufp = msgbuf;

putchar(c)		/* print a character on the console status driven */
	char c;
{
	register s;

	if (c != '\0' && c != '\r' && c != 0177) {
		*msgbufp++ = c;
		if (msgbufp >= &msgbuf[MSGBUFS])
			msgbufp = msgbuf;
	}
	s = splhigh();
	set(1, 0x12);		/* buffer won't empty unless we int */
	while ((SCC->a_rr0 & 04) == 0)
		;
	SCC->a_tdr = c;
	if (c == '\n')
		putchar('\r');
	set(1, reg1);
	splx(s);
}

ziinit()
{

	set(9, 0xC0);		/* hard reset */
	set(4, 0x4c);		/* /16, nopar, 1 stop */
	set(3, 0xC0);		/* 8bits */
	set(5, 0x62);		/* Xmit: 8bits, RTS  */
	set(9, 2);		/* no interrupts */
	set(11, 0x56);		/* use BRG */
	set(12, 14);		/* lower byte of time constant */
	set(13, 0);		/* high byte of time constant */
	set(14, 0x02);		/* BRG gen src = PCLK pin */
	set(14, 0x82);		/* clk = br gen */
	set(14, 0x03);		/* br generator enable */
	set(3, 0xC1);		/* rx enable */
	set(5, 0x6a);		/* tx enable */
	set(15, 0);		/* int on break, dcd */
	set(0, 0x10);		/* reset external/status */
	set(0, 0x10);		/* twice */
	set(2,0);
	set(9, 0);		/* no interrupts */
	set(5, 0xea);		/* turn on the dtr bit */
	{ int i; for (i = 10000; i--; i * i);}	/* wait for chip to setup */
}

