/* ttyld.c: Tty line discipline */
#include "param.h"
#include "types.h"
#include "errno.h"
#include "signal.h"
#include "stream.h"
#include "ttyio.h"

#define	NTTY_LDS	32
#define	CEOT		04
#define	TTYHOG		256
#define	CANBSIZ		256

extern	char	partab[];
static	char canonb[CANBSIZ];

static
char	maptab[] ={
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,'|',000,000,000,000,000,'`',
	'{','}',000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,000,000,
	000,000,000,000,000,000,'~',000,
	000,'A','B','C','D','E','F','G',
	'H','I','J','K','L','M','N','O',
	'P','Q','R','S','T','U','V','W',
	'X','Y','Z',000,000,000,000,000,
};
static	ttupput(), ttupsrv(), ttqopen(), ttqclose(), ttdnsrv();
static	ttdnput();
extern	nulldev();

struct	qinit	ttqinit[] = {
	{ ttupput, ttupsrv, ttqopen, ttqclose, 512, 50 },
	{ ttdnput, ttdnsrv, nulldev, nulldev, 512, 50 }
};

static	Tty	tty[NTTY_LDS];


static	struct	tchars	def_tc = { 0177, 034, 021, 023, 4, 0377 };


static
ttqopen(dev, q)		/* init an instance of a tty */
	dev_t dev;
	register struct queue *q;
{
	register Tty *tp;

	for (tp = tty; tp < &tty[NTTY_LDS]; tp++)
		if (tp->t_outq == NULL) {
			tp->t_erase = '#';
			tp->t_kill = '@';
			tp->t_col = 0;
			tp->t_delct = 0;
			tp->t_flags = ECHO|CRMOD;
			tp->tc = def_tc;
			tp->t_rawq = q;
			tp->t_outq = OTHERQ(q);
			q->ptr = (char *)tp;
			OTHERQ(q)->ptr = (char *)tp;
			q->flag |= QNOENB;
			return 0;
		}
	return ENOTTY;
}

static
ttqclose(dev, q)		/* free tty stuff for this queue */
	dev_t dev;
	register struct queue *q;
{
	register Tty *tp;

	tp = (Tty *)q->ptr;
	flushq(q);
	flushq(OTHERQ(q));
	tp->t_outq = NULL;
	tp->t_rawq = NULL;
	return 0;
}


static
ttupput(q, bp)		/* incomming characters */
	register struct queue *q;
	register struct block *bp;
{
	if (bp->type != M_DATA) {
		if (q->first == NULL)	/* q is marked no enable */
			qenable(q);
		putq(q, bp);
		return;
	}
	while (bp->rptr < bp->wptr)
		ttyinput(*bp->rptr++, (Tty *)q->ptr);
	freeb(bp);
}

static
ttupsrv(q)		/* service stuff in queue */
	register struct queue *q;
{
	register Tty *tp;

	tp = (Tty *)q->ptr;
	for (;;) {
		if (q->first == NULL)
			break;
		if (q->first->type != M_DATA) {
			(*q->next->qinfo->putp)(q->next, getq(q));
		if (q->next->flag & QFULL)
			break;
		if ((tp->t_flags & (RAW|CBREAK)) == 0 && tp->t_delct == 0)
			break;
			continue;
		}
		canon(tp);
	}
        if (q->first)   /* couldn't fit it all */
                q->next->flag |= QWANTW;
}

/*
 * Place a character on raw TTY input queue,
 * and scheduling srvice procedure as needed.
 * Also echo if required.
 * The arguments are the character and the appropriate
 * tty structure.
 */
static
ttyinput(c, tp)
	register c;
	register Tty *tp;
{
	register int t_flags;

	c &= 0377;
	t_flags = tp->t_flags;
#ifdef notdef
	if (t_flags&TANDEM)
		ttyblock(tp);
#endif
	if ((t_flags&RAW)==0) {
		c &= 0177;
		if (tp->t_state&TTSTOP) {
			if (c==tp->tc.t_startc) {
				tp->t_state &= ~TTSTOP;
				ttstart(tp);
				return;
			}
			if (c==tp->tc.t_stopc)
				return;
			tp->t_state &= ~TTSTOP;
			ttstart(tp);
		} else {
			if (c==tp->tc.t_stopc) {
				tp->t_state |= TTSTOP;
				ttstop(tp);
				return;
			}
			if (c==tp->tc.t_startc)
				return;
		}
		if (c==tp->tc.t_quitc || c==tp->tc.t_intrc) {
			flushq(tp->t_outq);
			flushq(tp->t_rawq);
			c = (c==tp->tc.t_intrc) ? SIGINT:SIGQUIT;
			ttsignal(tp, c);
			return;
		}
		if (c=='\r' && t_flags&CRMOD)
			c = '\n';
	}
	if (t_flags&LCASE && c>='A' && c<='Z')
		c += 'a'-'A';
	putc(c, tp->t_rawq);
	if(t_flags&(RAW|CBREAK)
		||(c=='\n'||c==tp->tc.t_eofc||c==tp->tc.t_brkc)){
		if ((t_flags & (RAW|CBREAK)) == 0) {
			putc(0377, tp->t_rawq);
			tp->t_delct++;
		}
		qenable(tp->t_rawq);
	}
	if (t_flags&ECHO) {
		if (c==tp->t_kill && (t_flags&(RAW|CBREAK))==0) {
				ttyoutput(c, tp);
				ttyoutput('\n', tp);
		} else if (c == 0177)
			ttyoutput('?', tp);
		else
			ttyoutput(c, tp);
	}
}

/*
 * transfer raw input list to canonical list,
 * doing erase-kill processing and handling escapes.
 */
static
canon(tp)
	register Tty *tp;
{
	register char *bp;
	char *bp1;
	register int c;
	int mc;

loop:
	bp = &canonb[2];
	while ((c=getc(tp->t_rawq)) >= 0) {
		if ((tp->t_flags&(RAW|CBREAK))==0) {
			if (c==0377) {
				tp->t_delct--;
				break;
			}
			if (bp[-1]!='\\') {
				if (c==tp->t_erase) {
					if (bp > &canonb[2])
						bp--;
					continue;
				}
				if (c==tp->t_kill)
					goto loop;
				if (c == tp->tc.t_eofc)
					continue;
			} else {
				mc = maptab[c];
				if (c==tp->t_erase || c==tp->t_kill)
					mc = c;
				if (mc && (mc==c || (tp->t_flags&LCASE))) {
					if (bp[-2] != '\\')
						c = mc;
					bp--;
				}
			}
		}
		*bp++ = c;
		if (bp>=canonb+CANBSIZ)
			break;
	}
	bp1 = &canonb[2];
	b_to_q(bp1, bp-bp1, tp->t_rawq->next);
#ifdef notdef
	if (tp->t_state&TBLOCK && tp->t_rawq->count < TTLOWAT) {
		putc(tp->tc.t_startc, tp->t_outq);
		tp->t_state &= ~TBLOCK;
	}
#endif
}

static
b_to_q(from, size, q)	/* stick on upstream queue */
	register char *from;
	register size;
	register struct queue *q;
{
	register struct block *bp;
	register n;

	do {
		bp = allocb(64);
		n = min(size, bp->lim - bp->base);
		bcopy(from, bp->wptr, n);
		bp->wptr += n;
		from += n;
		size -= n;
		if (size == 0)
			bp->class |= S_DELIM;
		(*q->qinfo->putp)(q, bp);
	} while (size > 0);
}

static
ttstart(tp)		/* send a start message */
	register Tty *tp;
{
	register struct block *bp;

	bp = allocb(0);
	bp->type = M_START;
	(*tp->t_outq->next->qinfo->putp)(tp->t_outq->next, bp);
}

static
ttstop(tp)		/* send a stop message */
	register Tty *tp;
{
	register struct block *bp;

	bp = allocb(0);
	bp->type = M_STOP;
	(*tp->t_outq->next->qinfo->putp)(tp->t_outq->next, bp);
}

static
ttsignal(tp, sig)		/* make a signal message and send it */
	register Tty *tp;
	register sig;
{
	register struct block *bp;

	bp = allocb(1);
	bp->type = M_SIGNAL;
	*bp->wptr++ = (char) sig;
	(*tp->t_rawq->next->qinfo->putp)(tp->t_rawq->next, bp);
}

static
ttdnsrv(q)		/* output processing */
	register struct queue *q;
{
	register Tty *tp;
	register c;
	register struct block *bp;
	register struct tchars *tcp;
	register struct sgttyb *sgp;

	tp = (Tty *)q->ptr;
	while (q->first && (q->next->flag & QFULL) == 0) {
		bp = getq(q);
		if (bp->type == M_IOCTL)
		switch (stiocom(bp)) {
		case TIOCSETC:
			bcopy(stiodata(bp), (char *)&tp->tc, sizeof tp->tc);
			bp->type = M_IOCACK;
			bp->wptr = bp->rptr;
			qreply(q, bp);
			qenable(q);
			break;
		case TIOCGETC:
			bcopy((char *)&tp->tc, bp->rptr, sizeof tp->tc);
			bp->type = M_IOCACK;
			bp->wptr = bp->rptr + sizeof tp->tc;
			qreply(q, bp);
			qenable(q);
			break;
		case TIOCGETP:
			sgp = (struct sgttyb *)(bp->rptr);
			sgp->sg_erase = tp->t_erase;
			sgp->sg_kill = tp->t_kill;
			sgp->sg_ispeed = sgp->sg_ospeed = 0;
			sgp->sg_flags = tp->t_flags;
			bp->type = M_IOCACK;
			bp->wptr = bp->rptr + sizeof (struct sgttyb);
			qreply(q, bp);
			qenable(q);
			break;
		case TIOCSETP:
			sgp = (struct sgttyb *)stiodata(bp);
			tp->t_erase = sgp->sg_erase;
			tp->t_kill = sgp->sg_kill;
			tp->t_flags = sgp->sg_flags;
			bp->type = M_IOCACK;
			bp->wptr = bp->rptr;
			qreply(q, bp);
			qenable(q);
			break;
		default:
			(*q->next->qinfo->putp)(q->next, bp);
			break;
		} else
			(*q->next->qinfo->putp)(q->next, bp);
	}
	if (q->first)	/* couldn't fit it all */
		q->next->flag |= QWANTW;
}

static
ttdnput(q, bp)		/* place character on output queue */
	register struct queue *q;
	register struct block *bp;
{
	if (bp->type != M_DATA) {
		putq(q, bp);
		return;
	}
	while (bp->rptr < bp->wptr)
		ttyoutput(*bp->rptr++, (Tty *)q->ptr);
	freeb(bp);
}


/*
 * put character on TTY output queue, adding delays,
 * expanding tabs, and handling the CR/NL bit.
 * It is called both from the top half for output, and from
 * interrupt level for echoing.
 * The arguments are the character and the tty structure.
 */
static
ttyoutput(c, tp)
register c;
register struct tty *tp;
{
	register char *colp;
	register ctype;

	if ((tp->t_flags & RAW) == 0) {
		c &= 0177;
		if (c == CEOT)
			return;
	} else {
		putc(c, tp->t_outq);
		return;
	}
	/* Turn tabs to spaces as required */
	if (c=='\t' && (tp->t_flags&TBDELAY)==XTABS) {
		c = 8;
		do
			ttyoutput(' ', tp);
		while (--c >= 0 && tp->t_col&07);
		return;
	}
	/*
	 * for upper-case-only terminals,
	 * generate escapes.
	 */
	if (tp->t_flags&LCASE) {
		colp = "({)}!|^~'`";
		while(*colp++)
			if(c == *colp++) {
				ttyoutput('\\', tp);
				c = colp[-2];
				break;
			}
		if ('a'<=c && c<='z')
			c += 'A' - 'a';
	}
	/*
	 * turn <nl> to <cr><lf> if desired.
	 */
	if (c=='\n' && tp->t_flags&CRMOD)
		ttyoutput('\r', tp);
	putc(c, tp->t_outq);
	/*
	 * Calculate delays.
	 * The numbers here represent clock ticks
	 * and are not necessarily optimal for all terminals.
	 * The delays are indicated by characters above 0200.
	 * In raw mode there are no delays and the
	 * transmission path is 8 bits wide.
	 */
	colp = &tp->t_col;
	ctype = partab[c];
	c = 0;
	switch (ctype&077) {

	/* ordinary */
	case 0:
		(*colp)++;

	/* non-printing */
	case 1:
		break;

	/* backspace */
	case 2:
		if (*colp)
			(*colp)--;
		break;

	/* newline */
	case 3:
		ctype = (tp->t_flags >> 8) & 03;
		if(ctype == 1) { /* tty 37 */
			if (*colp)
				c = max(((unsigned)*colp>>4) + 3, (unsigned)6);
		} else
		if(ctype == 2) /* vt05 */
			c = 6;
		*colp = 0;
		break;

	/* tab */
	case 4:
		ctype = (tp->t_flags >> 10) & 03;
		if(ctype == 1) { /* tty 37 */
			c = 1 - (*colp | ~07);
			if(c < 5)
				c = 0;
		}
		*colp |= 07;
		(*colp)++;
		break;

	/* vertical motion */
	case 5:
		if(tp->t_flags & VTDELAY) /* tty 37 */
			c = 0177;
		break;

	/* carriage return */
	case 6:
		ctype = (tp->t_flags >> 12) & 03;
		if(ctype == 1) /* tn 300 */
			c = 5;
		else if(ctype == 2) /* ti 700 */
			c = 10;
		*colp = 0;
	}
	if (c) {
		register struct block *bp;

		bp = allocb(1);
		bp->type = M_DELAY;
		*bp->wptr++ = c;
		putq(tp->t_outq, bp);	/* just stick on my queue */
	}
}

/*
 * when putc gets a new block it marks it as a delimiter.
 * On the way down the device doesn't care, and on the way up
 * they will be freed anyway.
 * This allows pipe to handle things in
 * units of blocks.
 */
static
putc(c, q)	/* put character on output */
	register c;
	register struct queue *q;
{
	register struct block *bp;
	register s;

	s = splnet();
	if ((bp = q->last) == NULL || bp->wptr >= bp->lim) {
		bp = allocb(64);
		bp->class |= S_DELIM;
		putq(q, bp);
	}
	*bp->wptr++ = c;
	splx(s);
}

static
getc(q)		/* pull a character from a queue */
	register struct queue *q;
{
	register struct block *bp;
	register s, c;

	s = splnet();
	if ((bp = q->first) == NULL) {
		splx(s);
		return -1;
	}
	c = *bp->rptr++;
	if (bp->rptr >= bp->wptr)
		freeb(getq(q));
	splx(s);
	return c;
}

