/*	@(#)subr.c	1.2 89/01/08 NFS Rev 2 Testsuite	*/
/*
 * Useful subroutines shared by all tests
 */

#include "tests.h"

char *Myname;

#ifdef ANSI

int unix_chdir(char * path);



#ifdef DOS
char *getwd(char * path);
#endif

#endif

/*
 * Build a directory tree "lev" levels deep
 * with "files" number of files in each directory
 * and "dirs" fan out.  Starts at the current directory.
 * "fname" and "dname" are the base of the names used for
 * files and directories.
 */
void
dirtree(lev, files, dirs, fname, dname, totfiles, totdirs)
	int lev;
	int files;
	int dirs;
	char *fname;
	char *dname;
	int *totfiles;
	int *totdirs;
{
	int fd;
	int f, d;
	char name[MAXPATHLEN];

	if (lev-- == 0) {
		return;
	}
	for ( f = 0; f < files; f++) {
		sprintf(name, "%s%d", fname, f);
		if ((fd = creat(name, CHMOD_YES)) < 0) {
			error("creat %s failed", name);
			exit(1);
		}
		(*totfiles)++;
		if (close(fd) < 0) {
			error("close %d failed", fd);
			exit(1);
		}
	}
	for ( d = 0; d < dirs; d++) {
		sprintf(name, "%s%d", dname, d);
		#ifdef DOS
		if (mkdir(name) < 0) {
		#else
		if (mkdir(name, 0777) < 0) {
		#endif
			error("mkdir %s failed", name);
			exit(1);
		}
		(*totdirs)++;
		if (unix_chdir(name) < 0) {
			error("chdir %s failed", name);
			exit(1);
		}
		dirtree(lev, files, dirs, fname, dname, totfiles, totdirs);
		if (unix_chdir("..") < 0) {
			error("chdir .. failed");
			exit(1);
		}
	}
}

/*
 * Remove a directory tree starting at the current directory.
 * "fname" and "dname" are the base of the names used for
 * files and directories to be removed - don't remove anything else!
 * "files" and "dirs" are used with fname and dname to generate
 * the file names to remove.
 *
 * This routine will fail if, say after removing known files,
 * the directory is not empty.
 *
 * This is used to test the unlink function and to clean up after tests.
 */
void
rmdirtree(lev, files, dirs, fname, dname, totfiles, totdirs, ignore)
	int lev;
	int files;
	int dirs;
	char *fname;
	char *dname;
	int *totfiles;		/* total removed */
	int *totdirs;		/* total removed */
	int ignore;
{
	int f, d;
	char name[MAXPATHLEN];

	if (lev-- == 0) {
		return;
	}
	for ( f = 0; f < files; f++) {
		sprintf(name, "%s%d", fname, f);
		if (unlink(name) < 0 && !ignore) {
			error("unlink %s failed", name);
			exit(1);
		}
		(*totfiles)++;
	}
	for ( d = 0; d < dirs; d++) {
		sprintf(name, "%s%d", dname, d);
		if (unix_chdir(name) < 0) {
			if (ignore)
				continue;
			error("chdir %s failed", name);
			exit(1);
		}
		rmdirtree(lev, files, dirs, fname, dname, totfiles, totdirs, ignore);
		if (unix_chdir("..") < 0) {
			error("chdir .. failed");
			exit(1);
		}
		if (rmdir(name) < 0) {
			error("rmdir %s failed", name);
			exit(1);
		}
		(*totdirs)++;
	}
}

/* VARARGS */
void
error(str, ar1, ar2, ar3, ar4, ar5, ar6, ar7, ar8, ar9)
	char *str;
	long ar1, ar2, ar3, ar4, ar5, ar6, ar7, ar8, ar9;
{
	char *ret;

	char path[MAXPATHLEN];

	if ((ret = getwd(path)) == NULL)

		fprintf(stderr, "%s: getwd failed\n", Myname);
	else
		fprintf(stderr, "\t%s: (%s) ", Myname, path);

	fprintf(stderr, str, ar1, ar2, ar3, ar4, ar5, ar6, ar7, ar8, ar9);
	if (errno)
		perror(" ");
	else
		fprintf(stderr, "\n");
	fflush(stderr);
	if (ret == NULL)
		exit(1);
}

static struct timeval ts, te;

/*
 * save current time in struct ts
 */
void
starttime()
{
	gettimeofday(&ts, (struct timezone *)0);
}

/*
 * sets the struct tv to the difference in time between
 * current time and the time in struct ts.
 */
void
endtime(tv)
	struct timeval *tv;
{

	gettimeofday(&te, (struct timezone *)0);
	if (te.tv_usec < ts.tv_usec) {
		te.tv_sec--;
		te.tv_usec += 1000000;
	}
	tv->tv_usec = te.tv_usec - ts.tv_usec;
	tv->tv_sec = te.tv_sec - ts.tv_sec;
}

void
printtimes(tv, nbytes)
	struct timeval *tv;	/* contains the elapsed time */
	long nbytes;		/* size * count */
{
	fprintf(stdout, " in %ld.%-2ld seconds",
		tv->tv_sec, tv->tv_usec / 10000L);
	if (nbytes > 0 && tv->tv_sec != 0)
		fprintf(stdout, " (%ld bytes/sec)", nbytes/tv->tv_sec);
}

/*
 * Set up and move to a test directory
 */
void
testdir(dir)
char *dir;
{
	struct stat statb;
	char str[MAXPATHLEN];
	char *getenv();

	/*
	 *  If dir is non-NULL, use that dir.  If NULL, first
	 *  check for env variable NFSTESTDIR.  If that is not
	 *  set, use the compiled-in TESTDIR.
	 */
	if (dir == NULL)
		if ((dir = getenv("NFSTESTDIR")) == NULL)
			dir = TESTDIR;

	if (stat(dir, &statb) == 0) {
		sprintf(str, "rm -r %s", dir);
		if (system(str) != 0) {
			error("can't remove old test directory %s", dir);
			exit(1);
		}
	}

	#ifdef DOS
	if (mkdir(dir) < 0) {
	#else
	if (mkdir(dir, 0777) < 0) {
	#endif
		error("can't create test directory %s", dir);
		exit(1);
	}
	if (unix_chdir(dir) < 0) {
		error("can't chdir to test directory %s", dir);
		exit(1);
	}
}

/*
 * Move to a test directory
 */
int
mtestdir(dir)
char *dir;
{
	char *getenv();

	/*
	 *  If dir is non-NULL, use that dir.  If NULL, first
	 *  check for env variable NFSTESTDIR.  If that is not
	 *  set, use the compiled-in TESTDIR.
	 */
	if (dir == NULL)
		if ((dir = getenv("NFSTESTDIR")) == NULL)
			dir = TESTDIR;

	if (unix_chdir(dir) < 0) {
		error("can't chdir to test directory %s", dir);
		return -1;
	}
	return 0;
}

/*
 *  get parameter at parm, convert to int, and make sure that
 *  it is at least min.
 */
long
getparm(parm, min, label)
char *parm, *label;
long min;
{
	long val = atol(parm);
	if (val < min) {
		error("Illegal %s parameter %ld, must be at least %ld",
			label, val, min);
		exit(1);
	}
	return val;
}

#ifdef DOS

#ifdef ANSI
void chdrive(char * path);
#endif

/*
 * Change to drive specified in path
 */

void
chdrive(path)
char *path;
{
	int desireddrive, drive;
	if (path[1] == ':')
		{
		desireddrive = toupper(path[0]) - ('A' - 1);
		_dos_setdrive(desireddrive, &drive);
		_dos_getdrive(&drive);
		if (drive != desireddrive)
			{
			error("can't change to drive %c:", path[0]);
			exit(1);
			}
		}
}

/*
 *  exit point for successful test
 */
void
complete()
{
	fprintf(stdout, "\t%s ok.\n", Myname);
#ifdef DOS
	chdrive(Myname);
#endif
	exit(0);
}


int
unix_chdir(path)
char *path;
{
	#ifdef DOS
	chdrive(path);
	#endif
	return chdir(path);
}

char *
getwd(path)
char * path;
{
	return getcwd(path, MAXPATHLEN);
}

int
unix_chmod(path, mode)
char *path;
int mode;
{
	int dosmode = (mode&0500 ? S_IREAD : 0) | (mode&0200 ? S_IWRITE : 0);
	return chmod(path, dosmode);
}

int
lstat(path, buf)
char *path;
struct stat *buf;
{
	return stat(path, buf);
}

void
gettimeofday(struct timeval *TV, struct timezone *TimeZone)
{
	struct dostime_t dostime;
	_dos_gettime(&dostime);
	TV->tv_sec = dostime.hour * 3600L
		+ dostime.minute * 60L
		+ dostime.second;
	TV->tv_usec = dostime.hsecond * 10000L;
	TimeZone = TimeZone;	/* shut up compiler/lint */
}

int
statfs(path, buf)
char *path;
struct statfs *buf;
{
	char *p = (char *) buf;
	int i;
	unsigned drive;
	struct diskfree_t diskspace;
	
	for (i = 0; i < sizeof(*buf); i++)
		*p++ = (char) -1;
	buf->f_type = 0;	/* that's what the man page says */
	if (path[1] == ':')
		drive = toupper(path[0]) - ('A' - 1);
	else
		_dos_getdrive(&drive);
	if (_dos_getdiskfree(drive, &diskspace))
		return -1;
	buf->f_bsize = diskspace.bytes_per_sector;
	buf->f_blocks = (long) diskspace.total_clusters
		* diskspace.sectors_per_cluster;
	buf->f_bfree = (long) diskspace.avail_clusters
		* diskspace.sectors_per_cluster;
	buf->f_bavail = buf->f_bfree;
	return 0;
}

/***************************************************************
DIRENT EMULATION FOR DOS
***************************************************************/
char pattern[MAXNAMLEN];
struct find_t findtst;
int maxentry;
int currententry;
int diropen = 0;
struct dirent *dirlist;
DIR dirst;

#ifdef ANSI
static void copynametolower(char *dest, char *src);
static void findt_to_dirent(struct find_t *f, struct dirent *d);
#endif

DIR *
opendir(dirname)
char *dirname;
{
	int i;
	unsigned attributes = _A_NORMAL|_A_RDONLY|_A_HIDDEN|_A_SUBDIR;
	strcpy(pattern, dirname);
	strcat(pattern, "\\*.*");
	if (diropen)
		return NULL;
	diropen = 1;
	dirlist = (struct dirent *) malloc(512 * sizeof(struct dirent));
	if (dirlist == NULL)
		return NULL;
	if (_dos_findfirst(pattern, attributes, &findtst))
		return NULL;
	findt_to_dirent(&findtst, &dirlist[0]);
	for (i = 1; ! _dos_findnext(&findtst); i++) {
		findt_to_dirent(&findtst, &dirlist[i]);
	}
	maxentry = i - 1;
	currententry = 0;
	return &dirst;
}

void
rewinddir(dirp)
DIR *dirp;
{
	int i;
	unsigned attributes = _A_NORMAL|_A_RDONLY|_A_HIDDEN|_A_SUBDIR;
	dirp = dirp;	/* shut up compiler */
	if (_dos_findfirst(pattern, attributes, &findtst)) {
		error("rewind failed");
		exit(1);
	}
	findt_to_dirent(&findtst, &dirlist[0]);
	for (i = 1; ! _dos_findnext(&findtst); i++) {
		findt_to_dirent(&findtst, &dirlist[i]);
	}
	maxentry = i - 1;
	currententry = 0;
}

long
telldir(dirp)
DIR *dirp;
{
	dirp = dirp;	/* keep compiler happy */
	return (long) currententry;
}

void
seekdir(dirp, loc)
DIR *dirp;
long loc;
{
	dirp = dirp;	/* keep compiler happy */
	if (loc <= (long) maxentry)
		currententry = (int) loc;
	/* else seekdir silently fails */
}

struct dirent *
readdir(dirp)
DIR *dirp;
{
	dirp = dirp;	/* shut up compiler */
	if (currententry > maxentry)
		return (struct dirent *) NULL;
	else {
		return &dirlist[currententry++];
	}
}

void
findt_to_dirent(f, d)
struct find_t *f;
struct dirent *d;
{
	copynametolower(d->d_name, f->name);
}

static void
copynametolower(dest, src)
char *dest;
char *src;
{
	int i;
	for (i = 0; dest[i] = (char) tolower((int) src[i]); i++) {
		/* null body */
	}
}

void
closedir(dirp)
DIR *dirp;
{
	dirp = dirp;	/* keep compiler happy */
	diropen = 0;
}

#endif /* DOS */

