#include <param.h>
#include <systm.h>
#include <mount.h>
#include <filsys.h>
#include <fblk.h>
#include <conf.h>
#include <buf.h>
#include <inode.h>
#include <ino.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <seg.h>
#include <errno.h>

struct fblk fblk;
extern struct inode inode[];
extern struct user u;
extern char b[];
extern struct mount mount[];

/*
 * alloc will obtain the next available
 * free disk block from the free list of
 * the specified device.
 * The super block has up to NICFREE remembered
 * free blocks; the last of these is read to
 * obtain NICFREE more . . .
 *
 * no space on dev x/y -- when
 * the free list is exhausted.
 */
struct buf *
alloc(dev)
dev_t dev;
{
	daddr_t bno;
	register struct filsys *fp;
	register struct buf *bp;
	int ka5sav;

	if (dlimit(1))
		return(NULL);
	ka5sav = ka5->r[0];
	fp = getfs(dev);
	while(fp->s_flock & S_BUSY)
	{
		fp->s_flock |= S_WANTED;
		sleep((caddr_t)&fp->s_flock, PINOD + 1);
	}
	do
	{
		if(fp->s_nfree <= 0)
			goto nospace;
		if (fp->s_nfree > NICFREE)
		{
			prdev("Bad free count", dev);
			goto nospace;
		}
		bno = fp->s_free[--fp->s_nfree];
		if(bno == 0)
			goto nospace;
	} while (badblock(fp, bno, dev));
	if(fp->s_nfree <= 0)
	{
		int ka5sav2;

		ka5sav2 = ka5->r[0];
		fp->s_flock |= S_BUSY;
		bp = bread(dev, bno);
		if ((bp->b_flags & B_ERROR) == 0)
		{
			ka5->r[0] = baddr(bp);
			bcopy(&b, &fblk, sizeof(struct fblk));
			ka5->r[0] = ka5sav2;
			fp->s_nfree = fblk.df_nfree;
			bcopy(fblk.df_free, fp->s_free, sizeof(fp->s_free));
		}
		else
			ka5->r[0] = ka5sav2;
		brelse(bp);
		if (fp->s_flock & S_WANTED)
			wakeup((caddr_t)&fp->s_flock);
		fp->s_flock = 0;
		if (fp->s_nfree <=0)
			goto nospace;
	}
	bp = getblk(dev, bno);
	clrbuf(bp);
	if (fp->s_tfree)
		fp->s_tfree--;
	fp->s_fmod = 1;
	ka5->r[0] = ka5sav;
	return(bp);

nospace:
	fp->s_nfree = 0;
	fp->s_tfree = 0;
	sleep((caddr_t)&lbolt, PZERO-1);	/* not interruptible */
	prdev("no space", dev);
	uprints("\nWrite - Device full\007\n");
	u.u_error = ENOSPC;
	ka5->r[0] = ka5sav;
	return(NULL);
}

/*
 * place the specified disk block
 * back on the free list of the
 * specified device.
 */
free(dev, bno)
dev_t dev;
daddr_t bno;
{
	register struct filsys *fp;
	register struct buf *bp;
	int ka5sav;

	ka5sav = ka5->r[0];
	fp = getfs(dev);
	fp->s_fmod = 1;
	while(fp->s_flock & S_BUSY)
	{
		fp->s_flock |= S_WANTED;
		sleep((caddr_t)&fp->s_flock, PINOD);
	}
	if (badblock(fp, bno, dev))
	{
		ka5->r[0] = ka5sav;
		return;
	}
	if(fp->s_nfree <= 0)
	{
		fp->s_nfree = 1;
		fp->s_free[0] = 0;
	}
	if(fp->s_nfree >= NICFREE)
	{
		int ka5sav2;

		ka5sav2 = ka5->r[0];
		fp->s_flock |= S_BUSY;
		bp = getblk(dev, bno);
		fblk.df_nfree = fp->s_nfree;
		bcopy(fp->s_free, fblk.df_free, sizeof(fp->s_free));
		ka5->r[0] = baddr(bp);
		bcopy(&fblk, &b, sizeof(struct fblk));
		ka5->r[0] = ka5sav2;
		fp->s_nfree = 0;
		bwrite(bp);
		if (fp->s_flock & S_WANTED)
			wakeup((caddr_t)&fp->s_flock);
		fp->s_flock = 0;
	}
	fp->s_free[fp->s_nfree++] = bno;
	fp->s_tfree++;
	fp->s_fmod = 1;
	ka5->r[0] = ka5sav;
}

/*
 * Check that a block number is in the
 * range between the I list and the size
 * of the device.
 * This is used mainly to check that a
 * garbage file system has not been mounted.
 *
 * bad block on dev x/y -- not in range
 */
badblock(fp, bn, dev)
register struct filsys *fp;
daddr_t bn;
dev_t dev;
{
	if (bn < fp->s_isize || bn >= fp->s_fsize)
	{
		prdev("bad block", dev);
		return(1);
	}
	return(0);
}

/*
 * Allocate an unused I node
 * on the specified device.
 * Used with file creation.
 * The algorithm keeps up to
 * NICINOD spare I nodes in the
 * super block. When this runs out,
 * a linear search through the
 * I list is instituted to pick
 * up NICINOD more.
 */
struct inode *
ialloc(dev)
dev_t dev;
{
	register struct filsys *fp;
	register struct buf *bp;
	register struct inode *ip;
	int i;
	struct dinode *dp;
	ino_t ino;
	daddr_t adr;
	int ka5sav;

	fp = getfs(dev);
loop:
	while(fp->s_ilock & S_BUSY)
	{
		fp->s_ilock |= S_WANTED;
		sleep((caddr_t)&fp->s_ilock, PINOD + 1);
	}
	if(fp->s_ninode > 0)
	{
		ino = fp->s_inode[--fp->s_ninode];
		if (ino < ROOTINO)
			goto loop;
		ip = iget(dev, ino);
		if(ip == NULL)
			return(NULL);
		if(ip->i_mode == 0)
		{
			i = NADDR * 3;
			while (i--)
				ip->i_addr[i] = 0;
			if (fp->s_tinode)
				fp->s_tinode--;
			fp->s_fmod = 1;
			return(ip);
		}
		/*
		 * Inode was allocated after all.
		 * Look some more.
		 */
		iput(ip);
		goto loop;
	}
	fp->s_ilock |= S_BUSY;
	ka5sav = ka5->r[0];
	ino = 1;
	for(adr = SUPERB+1; adr < fp->s_isize; adr++)
	{
		bp = bread(dev, adr);
		if (bp->b_flags & B_ERROR)
		{
			brelse(bp);
			ino += INOPB;
			continue;
		}
		dp = &b;
		for(i=0; i<INOPB; i++)
		{
			ka5->r[0] = baddr(bp);
			if(dp->di_mode != 0)
				goto cont;
			for(ip = &inode[0]; ip < &inode[NINODE]; ip++)
				if(dev==ip->i_dev && ino==ip->i_number)
					goto cont;
			ka5->r[0] = ka5sav;
			fp->s_inode[fp->s_ninode++] = ino;
			if(fp->s_ninode >= NICINOD)
				break;
		cont:
			ino++;
			dp++;
		}
		brelse(bp);
		ka5->r[0] = ka5sav;
		if(fp->s_ninode >= NICINOD)
			break;
	}
	if (fp->s_ilock & S_WANTED)
		wakeup((caddr_t)&fp->s_ilock);
	fp->s_ilock = 0;
	if(fp->s_ninode > 0)
		goto loop;
	prdev("Out of inodes", dev);
	uprints("\nCreate - Out of inodes\n");
	u.u_error = ENOSPC;
	fp->s_tinode = 0;
	return(NULL);
}

/*
 * Free the specified I node
 * on the specified device.
 * The algorithm stores up
 * to NICINOD I nodes in the super
 * block and throws away any more.
 */
ifree(dev, ino)
dev_t dev;
ino_t ino;
{
	register struct filsys *fp;
	register ka5sav;

	ka5sav = ka5->r[0];
	fp = getfs(dev);
	while (fp->s_ilock & S_BUSY)
	{
		fp->s_ilock |= S_WANTED;
		sleep((caddr_t)&fp->s_ilock, PINOD);
	}
	if(fp->s_ninode < NICINOD)
	{
		fp->s_inode[fp->s_ninode++] = ino;
		fp->s_fmod = 1;
	}
	fp->s_tinode++;
	ka5->r[0] = ka5sav;
}

/*
 * getfs maps a device number into
 * a pointer to the incore super
 * block.
 * The algorithm is a linear
 * search through the mount table.
 * A consistency check of the
 * in core free-block and i-node
 * counts.
 *
 * bad count on dev x/y -- the count
 *	check failed. At this point, all
 *	the counts are zeroed which will
 *	almost certainly lead to "no space"
 *	diagnostic
 * panic: no fs -- the device is not mounted.
 *	this "cannot happen"
 */
struct filsys *
getfs(dev)
dev_t dev;
{
	register struct mount *mp;
	register struct filsys *fp;
	register struct buf *bp;

	for(mp = &mount[0]; mp < &mount[NMOUNT]; mp++)
		if ((bp = mp->m_bufp) != NULL && mp->m_dev == dev)
		{
			ka5->r[0] = baddr(bp);
			fp = &b;
			if(fp->s_nfree > NICFREE || fp->s_ninode > NICINOD)
			{
				prdev("bad count", dev);
				fp->s_nfree = 0;
				fp->s_ninode = 0;
			}
			return(fp);
		}
	panic("no fs");
}

/*
 * update is the internal name of
 * 'sync'. It goes through the disk
 * queues to initiate sandbagged IO;
 * goes through the I nodes to write
 * modified nodes; and it goes through
 * the mount table to initiate modified
 * super blocks.
 */
update()
{
	register struct inode *ip;
	register struct mount *mp;
	register struct buf *bp;
	struct filsys *fp;

	if(updlock)
		return;
	updlock++;
	for(mp = &mount[0]; mp < &mount[NMOUNT]; mp++)
		if ((bp = mp->m_bufp) != NULL)
		{
			ka5->r[0] = baddr(bp);
			fp = &b;
			if ((fp->s_fmod == 0 && mp->m_dev != rootdev) ||
			    fp->s_ilock!=0 || fp->s_flock!=0 || fp->s_ronly!=0)
				continue;
			bp = getblk(mp->m_dev, SUPERB);
			if (bp != mp->m_bufp)
				panic("update block");
			fp->s_fmod = 0;
			fp->s_time = time;
			bwrite(bp);
		}
	for(ip = &inode[0]; ip < &inode[NINODE]; ip++)
		if ((ip->i_flag & ILOCK) == 0 && ip->i_count)
		{
			ip->i_flag |= ILOCK;
			ip->i_count++;
			iupdat(ip, time, time);
			iput(ip);
		}
	updlock = 0;
	bflush(NODEV);
}
